#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <stdint.h>
#include <netdb.h>
#include <ctype.h>
#include <stdarg.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/socket.h>

#include "../include/yc_opt.h"
#include "../include/yc_basetype.h"
#include "../include/yc_debug.h"
#include "../include/yc_reg.h"
#include "../include/list.h"
#include "../include/yc_timer.h"
#include "../include/yc_rtsp.h"


#define RTSP_UDP    0
#define RTSP_ASF    1
#define RTSP_RTCP   0
#if 1
#define rtsp_printf(...)     printf("rtsp: " __VA_ARGS__)
#else
#define rtsp_printf(...)
#endif
#if 1
#define rtp_printf(...)     printf("rtp: " __VA_ARGS__)
#else
#define rtp_printf(...)
#endif
#define  rtsp_malloc    yc_malloc
#define  rtsp_free      yc_free

#define RTSP_CHANNEL_NUM_MAX            8/* audio vedio ts Ӧþ͹ */
#define RTSP_DEFAULT_BLOCK_SIZE         1600
#define RTSP_DEFAULT_ALIVE_TIME         60/*  */
#define RTSP_DEFAULT_MAXPS              1600

#define RTSP_ERRCODE_SUCCESS            0
#define RTSP_ERRCODE_GENERAL_FAILED    -1
#define RTSP_ERRCODE_SOCKET_CLOSE      -2
#define RTSP_ERRCODE_RECV_TIMEOUT      -3
#define RTSP_ERRCODE_RECV_FAILED       -4
#define RTSP_ERRCODE_USER_CANCEL       -5
#define RTSP_ERRCODE_STREAM_END        -6
#define RTSP_ERRCODE_RTCP_PACKET       -7
#define RTSP_ERRCODE_UNKNOWN_ID        -8
#define RTSP_ERRCODE_RTP_PRASE         -9

#if RTSP_RTCP
typedef struct RTPStatistics {
    uint16_t max_seq;           ///< highest sequence number seen
    uint32_t cycles;            ///< shifted count of sequence number cycles
    uint32_t base_seq;          ///< base sequence number
    uint32_t bad_seq;           ///< last bad sequence number + 1
    int probation;              ///< sequence packets till source is valid
    uint32_t received;          ///< packets received
    uint32_t expected_prior;    ///< packets expected in last interval
    uint32_t received_prior;    ///< packets received in last interval
    uint32_t transit;           ///< relative transit time for previous packet
    uint32_t jitter;            ///< estimated jitter.

    uint32_t ssrc;
    uint16_t curr_seq;
    uint64_t last_rtcp_ntp_time;
    int64_t last_rtcp_reception_time;
    uint32_t base_timestamp;
    uint64_t first_rtcp_ntp_time;

    unsigned int octet_count;
    unsigned int last_octet_count;
} RTPStatistics;
#endif

typedef enum rtsp_transport_mode
{
    RTSP_TRANSPORT_MODE_UDP = 0,
    RTSP_TRANSPORT_MODE_TCP,
    RTSP_TRANSPORT_MODE_UDP_MULTICAST
}rtsp_trans_mode_e;

typedef enum rtsp_play_state
{
    RTSP_STATE_STOP = 0,
    RTSP_STATE_PLAY,
    RTSP_STATE_PAUSE
}rtsp_state_e;

typedef enum rtsp_request_method
{
    RTSP_METHOD_OPTIONS = 0,
    RTSP_METHOD_DESCRIBE,
    RTSP_METHOD_SETUP,
    RTSP_METHOD_TEARDOWN,
    RTSP_METHOD_PLAY,
    RTSP_METHOD_PAUSE,
    RTSP_METHOD_GET_PARAMETER,
    RTSP_METHOD_SET_PARAMETER,
}rtsp_request_method_e;

struct tagRtsp_Prvi{
    char *server;
    char *title;
    char *url;
    char *session_id;
    rtsp_state_e state;
    unsigned int   cseq;
    unsigned int   rtpNumber;
	unsigned int   rtcpNumber;
    unsigned short block_size;
    unsigned short timeout;

    unsigned int  session_num;
    unsigned char stream_index;
    unsigned char rtp_channel[RTSP_CHANNEL_NUM_MAX];
    unsigned char rtcp_channel[RTSP_CHANNEL_NUM_MAX];

    rtsp_trans_mode_e tmode;

    int skt_num;
    int rtp_skt_num;
    int rtcp_skt_num;
    unsigned short rtp_local_port;
    unsigned short rtp_server_port;
    unsigned short rtcp_server_port;

    unsigned short start_seq;
#if RTSP_RTCP
    RTPStatistics  stat;
#endif
    unsigned int   rtptime;
    unsigned short scale;/* *1000 */

    unsigned long long begin_npt;/* ms */
    unsigned long long end_npt;

    unsigned short sdp_len;
    unsigned short maxps;
    unsigned char *asf_head;
    unsigned short asf_len;

    void *timer;
    pthread_rwlock_t stLock;
    struct list_head stNode;

    char *buffer;

    unsigned int reset_counter;
    unsigned int curr_npt;

    unsigned char server_method;/* per bit, 0-7 -> OPTIONS, DESCRIBE, SETUP, TEARDOWN, PLAY, PAUSE, GET_PARAMETER, SET_PARAMETER */
    unsigned char is_retry;
    unsigned char is_first_packet;
};

#define RTSP_ADDRESS_LEN            256
#define RTSP_CMD_BUF_SIZE_MAX       512
#define RTSP_READ_BUF_SIZE_MAX      1024
#define RTSP_RTP_DEFAULT_LOCAT_PORT 5004

static char* user_agent_str   = "User-Agent: yc rtsp client v1.0.0.910\r\n";
static char* rtsp_version_str = " RTSP/1.0\r\n";

#define RTP_RCV_WAIT_TIME       3/* seconds */
#define RTCP_PACKET_BUF_MAX     256

/* RTCP packet types */
enum RTCPType {
    RTCP_FIR    = 192,
    RTCP_NACK, // 193
    RTCP_SMPTETC,// 194
    RTCP_IJ,   // 195
    RTCP_SR     = 200,
    RTCP_RR,   // 201
    RTCP_SDES, // 202
    RTCP_BYE,  // 203
    RTCP_APP,  // 204
    RTCP_RTPFB,// 205
    RTCP_PSFB, // 206
    RTCP_XR,   // 207
    RTCP_AVB,  // 208
    RTCP_RSI,  // 209
    RTCP_TOKEN,// 210
};

/* RTCP packets use 0.5% of the bandwidth */
#define RTCP_TX_RATIO_NUM 5
#define RTCP_TX_RATIO_DEN 1000

#define RTP_RTCP_PACKET_VERSION          2
#define RTP_SEQ_MOD                     (1 << 16)
#define FFABS(a)                        ((a) >= 0 ? (a) : (-(a)))
#define FFMIN(a,b)                      ((a) > (b) ? (b) : (a))
#define FFMAX(a,b)                      ((a) > (b) ? (a) : (b))
#define RTP_PT_IS_RTCP(x)               (((x) >= RTCP_FIR && (x) <= RTCP_IJ) || \
                                        ((x) >= RTCP_SR  && (x) <= RTCP_TOKEN))
#define AV_NOPTS_VALUE                  (UINT64_C(0x8000000000000000))
#ifndef UINT64_C
#define UINT64_C(c)                     (c ## ULL)
#endif

#define RTCP_CNAME_STR                  "yc_rtsp_client"

struct rtcp_packet_head{
    unsigned char rc: 5;
    unsigned char pad: 1;
    unsigned char ver: 2;
    unsigned char pt;
    unsigned short length;
    unsigned int ssrc;
    unsigned int ssrc_1;
    unsigned int fraction; /* fraction 8bit, lost 24bit */
    unsigned int ext_high_seq;
    unsigned int jitter;
    unsigned int lsr;
    unsigned int dlsr;
};

struct rtp_cache_packet{
    struct list_head stNode;
    char *buf;
    unsigned short len;
    unsigned char stream_id;
};

#if RTSP_RTCP
static int rtsp_send_rtcp(yc_rtsp_s *ctx, unsigned char stream_id, unsigned short count)
{
    struct rtcp_packet_head *head;
    unsigned char offset = 0;
    int len;
    int ret = -1;
    unsigned char *rtcp_packet;
    char cname[16];
    unsigned char *pos = NULL;
    uint32_t lost;
    uint32_t extended_max;
    uint32_t expected_interval;
    uint32_t received_interval;
    int32_t  lost_interval;
    uint32_t expected;
    uint32_t fraction;
    int rtcp_bytes;

    /* TODO: I think this is way too often; RFC 1889 has algorithm for this */
    /* XXX: MPEG pts hardcoded. RTCP send every 0.5 seconds */
    ctx->prvi->stat.octet_count += count;
    rtcp_bytes = ((ctx->prvi->stat.octet_count - ctx->prvi->stat.last_octet_count) * RTCP_TX_RATIO_NUM) /
        RTCP_TX_RATIO_DEN;
    rtcp_bytes /= 50; // mmu_man: that's enough for me... VLC sends much less btw !?
    if (rtcp_bytes < 28)
        return -1;
    ctx->prvi->stat.last_octet_count = ctx->prvi->stat.octet_count;

    rtcp_packet = rtsp_malloc(RTCP_PACKET_BUF_MAX);
    if (!rtcp_packet)
        return -1;

    memset(rtcp_packet, 0, RTCP_PACKET_BUF_MAX);

    if (RTSP_TRANSPORT_MODE_TCP == ctx->prvi->tmode)
        offset = 4;

    head = (struct rtcp_packet_head *)(rtcp_packet + offset);

    head->ver = RTP_RTCP_PACKET_VERSION;
    head->pad = 0;
    head->rc  = 1;  /* 1 report block */
    head->pt  = 201;/* RR */
    head->length = htons(7);
    head->ssrc = htonl(ctx->prvi->stat.ssrc + 1);
    head->ssrc_1 = htonl(ctx->prvi->stat.ssrc);

    extended_max          = ctx->prvi->stat.cycles + ctx->prvi->stat.max_seq;
    expected              = extended_max - ctx->prvi->stat.base_seq;
    lost                  = expected - ctx->prvi->stat.received;
    lost                  = FFMIN(lost, 0xffffff); // clamp it since it's only 24 bits...
    expected_interval     = expected - ctx->prvi->stat.expected_prior;
    ctx->prvi->stat.expected_prior = expected;
    received_interval     = ctx->prvi->stat.received - ctx->prvi->stat.received_prior;
    ctx->prvi->stat.received_prior = ctx->prvi->stat.received;
    lost_interval         = expected_interval - received_interval;
    if (expected_interval == 0 || lost_interval <= 0)
        fraction = 0;
    else
        fraction = (lost_interval << 8) / expected_interval;

    fraction = (fraction << 24) | lost;/* lost percent/lost count */
    head->fraction = htonl(fraction);
    head->ext_high_seq = htonl(extended_max);
    head->jitter = htonl(ctx->prvi->stat.jitter >> 4);

    if (ctx->prvi->stat.last_rtcp_ntp_time == AV_NOPTS_VALUE) {
        head->lsr = 0;
        head->dlsr = 0;
    } else {
        uint32_t middle_32_bits   = ctx->prvi->stat.last_rtcp_ntp_time >> 16; // this is valid, right? do we need to handle 64 bit values special?
        uint32_t delay_since_last = 126543;//326543;/* need calculate */

        head->lsr = htonl(middle_32_bits);
        head->dlsr = htonl(delay_since_last);
    }

    pos = rtcp_packet + offset + sizeof(struct rtcp_packet_head);

    *pos++ = (RTP_RTCP_PACKET_VERSION << 6) + 1;
    *pos++ = RTCP_SDES;
    sprintf(cname, RTCP_CNAME_STR);
    len = strlen(cname);
    *pos++ = (((7 + len + 3) / 4) & 0xff00) >> 8;
    *pos++ =  ((7 + len + 3) / 4) & 0xff;
    *pos++ =  ((ctx->prvi->stat.ssrc + 1) & 0xff000000) >> 24;
    *pos++ =  ((ctx->prvi->stat.ssrc + 1) & 0xff0000)   >> 16;
    *pos++ =  ((ctx->prvi->stat.ssrc + 1) & 0xff00)     >> 8;
    *pos++ =   (ctx->prvi->stat.ssrc + 1) & 0xff;
    *pos++ = 0x01;
    *pos++ = len;
    memcpy(pos, cname, len);
    pos = pos + len;
    *pos++ = 0;

    for (len = (7 + len) % 4; len % 4; len++)
        *pos++ = 0;

    len = pos - (rtcp_packet + offset);

redo:
    if (RTSP_TRANSPORT_MODE_TCP == ctx->prvi->tmode)
    {
        rtcp_packet[0] = '$';
        rtcp_packet[1] = ctx->prvi->rtcp_channel[stream_id];
        rtcp_packet[2] = (len & 0xff00) >> 8;
        rtcp_packet[3] =  len & 0xff;
        ret = send(ctx->prvi->skt_num, rtcp_packet, len + offset, 0);
    }
    else //if (RTSP_TRANSPORT_MODE_UDP == ctx->prvi->tmode)
    {
        ret = send(ctx->prvi->rtcp_skt_num, rtcp_packet, len, 0);
    }

    if ((ret <= 0) && (errno != EINTR && errno != EWOULDBLOCK && errno != EAGAIN))
    {
        rtsp_printf("sent rtcp packet failed, socket error.\r\n");
        ret = RTSP_ERRCODE_GENERAL_FAILED;
    }
    else if ((ret < 0) && (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN))
    {
        rtsp_printf("sent rtcp packet failed, but continue...\r\n");
        goto redo;
    }
    else
        ret = 0;

    rtsp_free(rtcp_packet);

    return ret;
}
#endif

static int rtp_select_read(yc_rtsp_s *rtsp_ctx, void* buffer, unsigned short ReadSize)
{
    fd_set read_set;
    struct timeval tv;
    int ret;
    int result = 0;

redo1:
    if (RTSP_STATE_STOP == rtsp_ctx->prvi->state)
        return RTSP_ERRCODE_USER_CANCEL;

    FD_ZERO(&read_set);
    FD_SET(rtsp_ctx->prvi->skt_num, &read_set);
    tv.tv_sec  = RTP_RCV_WAIT_TIME;
    tv.tv_usec = 0;

    ret = select(rtsp_ctx->prvi->skt_num + 1, &read_set, NULL, NULL, &tv);
    if (ret > 0)
    {
        if (FD_ISSET(rtsp_ctx->prvi->skt_num, &read_set))
        {
            result = recv(rtsp_ctx->prvi->skt_num, buffer, ReadSize, 0);
            if (result == 0)
            {
                rtp_printf("rtsp conntection was closed.\n");
    			return RTSP_ERRCODE_SOCKET_CLOSE;
            }
    		else if (result < 0)
    		{
    			// error reading TCP socket
    			if ((errno == EINTR) || (errno == EWOULDBLOCK) || (errno == EAGAIN))
    			{
                    rtp_printf("error reading TCP socket, but continue...\n");
                    goto redo1;
    			}
    			else
    			{
    			    rtp_printf("error reading TCP socket\n");
    			    return RTSP_ERRCODE_RECV_FAILED;
    			}
    		}
    		FD_CLR(rtsp_ctx->prvi->skt_num, &read_set);
        }
        else
        {
            result = RTSP_ERRCODE_GENERAL_FAILED;/* exist situation? */
        }
    }
    else if (0 == ret)
    {
        result = RTSP_ERRCODE_RECV_TIMEOUT;
    }
    else
    {
        result = RTSP_ERRCODE_GENERAL_FAILED;
    }

    return result;
}

static int rtp_read_by_size(yc_rtsp_s *rtsp_ctx, char* buffer,
                            unsigned short bufferMaxSize,
                            unsigned short NextTCPReadSize)
{
	unsigned int totBytesToRead;
	int          curBytesToRead;
	unsigned int curBytesRead;

    if (bufferMaxSize < NextTCPReadSize)
    {
        rtp_printf("bufferMaxSize < NextTCPReadSize\n");
        return RTSP_ERRCODE_GENERAL_FAILED;
	}

    curBytesRead   = 0;
    totBytesToRead = NextTCPReadSize;

    do
    {
        curBytesToRead = rtp_select_read(rtsp_ctx, buffer + curBytesRead, totBytesToRead);
        if (0 >= curBytesToRead)
            return curBytesToRead;

        curBytesRead   += curBytesToRead;
        totBytesToRead -= curBytesToRead;
    } while(totBytesToRead > 0);

	return curBytesRead;
}

#if RTSP_RTCP
static int rtcp_read(rtsp_ctx_s *rtsp_ctx,
                     char *rcv_buf,
                     unsigned int ReadSize,
                     char **data_ptr,
                     unsigned short *data_size)
{
    int payload_len;
    int datalen = ReadSize;

    while (datalen >= 4) {
        payload_len = FFMIN(datalen, (ntohs(*(unsigned short *)(rcv_buf + 2)) + 1) * 4);

        switch ((unsigned char)rcv_buf[1]) {
        case RTCP_SR:
            if (payload_len < 20) {
                rtp_printf("Invalid length for RTCP SR packet\n");
                return RTSP_ERRCODE_GENERAL_FAILED;
            }

            rtsp_ctx->prvi->stat.last_rtcp_reception_time = 0;/* Ҫntp֧ */
            rtsp_ctx->prvi->stat.last_rtcp_ntp_time  = ntohs(*(unsigned short *)(rcv_buf + 8));

            if (rtsp_ctx->prvi->stat.first_rtcp_ntp_time == AV_NOPTS_VALUE) {
                rtsp_ctx->prvi->stat.first_rtcp_ntp_time = rtsp_ctx->prvi->stat.last_rtcp_ntp_time;
            }

            break;
        case RTCP_BYE:
            rtp_printf("rtcp bye: stream end\n");
            return RTSP_ERRCODE_STREAM_END;
        }

        rcv_buf += payload_len;
        datalen -= payload_len;
    }

    return RTSP_ERRCODE_RTCP_PACKET;
}

/*
 * Called whenever there is a large jump in sequence numbers,
 * or when they get out of probation...
 */
static void rtp_init_sequence(RTPStatistics *s, uint16_t seq)
{
    s->max_seq        = seq;
    s->cycles         = 0;
    s->base_seq       = seq - 1;
    s->bad_seq        = RTP_SEQ_MOD + 1;
    s->received       = 0;
    s->expected_prior = 0;
    s->received_prior = 0;
    s->jitter         = 0;
    s->transit        = 0;
}

/* Returns 1 if we should handle this packet. */
static int rtp_valid_packet_in_sequence(RTPStatistics *s, uint16_t seq)
{
    uint16_t udelta = seq - s->max_seq;
    const int MAX_DROPOUT    = 3000;
    const int MAX_MISORDER   = 100;
    const int MIN_SEQUENTIAL = 2;

    /* source not valid until MIN_SEQUENTIAL packets with sequence
     * seq. numbers have been received */
    if (s->probation) {
        if (seq == s->max_seq + 1) {
            s->probation--;
            s->max_seq = seq;
            if (s->probation == 0) {
                rtp_init_sequence(s, seq);
                s->received++;
                return 1;
            }
        } else {
            s->probation = MIN_SEQUENTIAL - 1;
            s->max_seq   = seq;
        }
    } else if (udelta < MAX_DROPOUT) {
        // in order, with permissible gap
        if (seq < s->max_seq) {
            // sequence number wrapped; count another 64k cycles
            s->cycles += RTP_SEQ_MOD;
        }
        s->max_seq = seq;
    } else if (udelta <= RTP_SEQ_MOD - MAX_MISORDER) {
        // sequence made a large jump...
        if (seq == s->bad_seq) {
            /* two sequential packets -- assume that the other side
             * restarted without telling us; just resync. */
            rtp_init_sequence(s, seq);
        } else {
            s->bad_seq = (seq + 1) & (RTP_SEQ_MOD - 1);
            return 0;
        }
    } else {
        // duplicate or reordered packet...
    }
    s->received++;
    return 1;
}

static void rtcp_update_jitter(RTPStatistics *s, uint32_t sent_timestamp,
                               uint32_t arrival_timestamp)
{
    // Most of this is pretty straight from RFC 3550 appendix A.8
    uint32_t transit = arrival_timestamp - sent_timestamp;
    uint32_t prev_transit = s->transit;
    int32_t d = transit - prev_transit;
    // Doing the FFABS() call directly on the "transit - prev_transit"
    // expression doesn't work, since it's an unsigned expression. Doing the
    // transit calculation in unsigned is desired though, since it most
    // probably will need to wrap around.
    d = FFABS(d);
    s->transit = transit;
    if (!prev_transit)
        return;
    s->jitter += d - (int32_t) ((s->jitter + 8) >> 4);
}
#endif

static void rtp_curr_npt(yc_rtsp_s *rtsp, yc_rtp_packet_s *rtp)
{
    if (0 == ntohl(rtp->timestamp))
        rtsp->prvi->reset_counter++;

    rtsp->prvi->curr_npt = (((double)rtsp->prvi->begin_npt / 1000) + ((0x100000000 * rtsp->prvi->reset_counter + ntohl(rtp->timestamp) - rtsp->prvi->rtptime) / rtsp->media->timestamp_frequency) * ((double)rtsp->prvi->scale / 1000)) * 1000;
    //rtp_printf("curr npt: %llu\r\n", wm_sm_rtp_curr_npt);

    return;
}

static int rtp_read(yc_rtsp_s *rtsp_ctx,
                    unsigned char streamChannelId,
                    char *rcv_buf,
                    unsigned int ReadSize,
                    char **data_ptr,
                    unsigned short *data_size,
                    yc_rtp_packet_s **rtp_head)
{
	int datasize;
	char *bufptr;
	unsigned char  pad_len;
	unsigned short seq;
	yc_rtp_packet_s *head;
	yc_rtp_ext_head_s *ext_head;

    datasize = ReadSize;

    if (datasize < sizeof(yc_rtp_packet_s))
    {
        rtp_printf("rtp packet head too short\n");
        return RTSP_ERRCODE_RTP_PRASE;
    }

    bufptr = rcv_buf;
    head = (yc_rtp_packet_s *)bufptr;

    if (head->version != RTP_RTCP_PACKET_VERSION)
    {
	    rtp_printf("rtp packet ver %u not is %u, ReadSize %hhu-%u\n", head->version, RTP_RTCP_PACKET_VERSION, streamChannelId, ReadSize);
	    return RTSP_ERRCODE_RTP_PRASE;
	}

    datasize -= sizeof(yc_rtp_packet_s);
    bufptr   += sizeof(yc_rtp_packet_s);

    if (head->csrc_count)
    {
        datasize -= head->csrc_count * 4;
        bufptr   += head->csrc_count * 4;
    }

    if (datasize <= 0)
    {
        rtp_printf("rtp packet csrc too long\n");
        return RTSP_ERRCODE_RTP_PRASE;
    }

    if (head->extension)
    {
        if (datasize < sizeof(yc_rtp_ext_head_s))
        {
            rtp_printf("rtp packet ext head too short\n");
            return RTSP_ERRCODE_RTP_PRASE;
        }

        ext_head = (yc_rtp_ext_head_s *)bufptr;

        datasize -= sizeof(yc_rtp_ext_head_s) + ntohs(ext_head->length) * 4;
        bufptr   += sizeof(yc_rtp_ext_head_s) + ntohs(ext_head->length) * 4;
    }

    if (datasize <= 0)
    {
        rtp_printf("rtp packet ext too long\n");
        return RTSP_ERRCODE_RTP_PRASE;
    }

    if (head->padding)
    {
        pad_len = rcv_buf[ReadSize - 1];

        if (datasize < pad_len)
        {
            rtp_printf("rtp packet padding len too long\n");
            return RTSP_ERRCODE_RTP_PRASE;
        }

        datasize -= pad_len;
    }

    seq = ntohs(head->seq_no);

    /* maybe send the first previ packet */
    if (rtsp_ctx->prvi->is_first_packet)
    {
        rtsp_ctx->prvi->is_first_packet = 0;
        if (seq < rtsp_ctx->prvi->start_seq)
        {
            rtp_printf("\r\n\r\n###############################\r\n"
               "start_seq %hu, seqno %hu\r\n###############################\r\n\r\n", rtsp_ctx->prvi->start_seq, seq);
            return RTSP_ERRCODE_RTP_PRASE;
        }
    }

#if RTSP_RTCP
    rtsp_ctx->prvi->stat.ssrc = ntohl(head->ssrc);

    // only do something with this if all the rtp checks pass...
    if (!rtp_valid_packet_in_sequence(&rtsp_ctx->prvi->stat, seq)) {
        rtp_printf("RTP: PT=%02x: bad cseq %04x expected=%04x\n",
               head->payload_type, ntohs(head->seq_no), ((rtsp_ctx->prvi->stat.curr_seq + 1) & 0xffff));
        return RTSP_ERRCODE_RTP_PRASE;
    }

    rtcp_update_jitter(&rtsp_ctx->prvi->stat, ntohl(head->timestamp), 1452256980);//3452256980);/* about sample */
    rtsp_ctx->prvi->stat.curr_seq = seq;
#endif

    rtp_curr_npt(rtsp_ctx, head);

    if (data_ptr)
        *data_ptr  = bufptr;
    if (data_size)
        *data_size = datasize;
    if (rtp_head)
        *rtp_head = head;

#if RTSP_RTCP
    if (RTSP_TRANSPORT_MODE_TCP != rtsp_ctx->prvi->tmode)
        rtsp_send_rtcp(rtsp_ctx, streamChannelId, ReadSize);
#endif

    return RTSP_ERRCODE_SUCCESS;
}

static int rtsp_read_packet(yc_rtsp_s *rtsp_ctx,
                            unsigned char streamChannelId,
                            unsigned int ReadSize,
                            char *rcv_buf,
                            unsigned short buf_size,
                            char **data_ptr,
                            unsigned short *data_size,
                            yc_rtp_packet_s **rtp_head)
{
    int i;
	int ret;
	int found = 0;

    ret = rtp_read_by_size(rtsp_ctx, rcv_buf, buf_size, ReadSize);
    if (0 >= ret)
    {
        rtp_printf("rtp_read_by_size failed %d\n", ret);
        return ret;
    }

#if RTSP_RTCP
    if (RTP_PT_IS_RTCP(rcv_buf[1])) {
        return rtcp_read(rtsp_ctx, rcv_buf, ReadSize, data_ptr, data_size);
    }
#endif

    for (i = 0; i < RTSP_CHANNEL_NUM_MAX; i++)
    {
        if (streamChannelId == rtsp_ctx->prvi->rtp_channel[i])
        {
            found = 1;
            ret = rtp_read(rtsp_ctx, streamChannelId, rcv_buf, ReadSize, data_ptr, data_size, rtp_head);
            break;
        }
#if RTSP_RTCP
        else if (streamChannelId == rtsp_ctx->prvi->rtcp_channel[i])
        {
            found = 1;
            ret = rtcp_read(rtsp_ctx, rcv_buf, ReadSize, data_ptr, data_size);
            break;
        }
#endif
    }

    if (!found)
    {
        if (1 != streamChannelId)
            rtp_printf("unkown channel id '%hhu', size '%u'\n", streamChannelId, ReadSize);
        ret = RTSP_ERRCODE_UNKNOWN_ID;/* up redo */
    }

	return ret;
}

static int rtsp_read_rtp_over_tcp(yc_rtsp_s *rtsp_ctx,
                                  char *rcv_buf,
                                  unsigned short buf_size,
                                  char **data_ptr,
                                  unsigned short *data_size,
                                  yc_rtp_packet_s **rtp_head)
{
    unsigned short size;
	unsigned char c;
	unsigned char streamChannelId;
	unsigned int ReadSize;
	int result = 0;

//redo:
	do
	{
        result = rtp_select_read(rtsp_ctx, &c, 1);
        if (0 >= result)
            return result;
	} while (c != '$');

    result = rtp_select_read(rtsp_ctx, &streamChannelId, 1);
    if (0 >= result)
        return result;

    //if (0 != streamChannelId)
    //rtp_printf("stream id: '%hhu'.\n", streamChannelId);

    result = rtp_select_read(rtsp_ctx, (unsigned char*)&size, 2);
    if (0 >= result)
        return result;

	ReadSize = ntohs(size);
	//if (0 != streamChannelId)
	//rtp_printf("stream size '%hu'\n", ReadSize);

	if (buf_size < ReadSize)
	    ReadSize = buf_size;

	result = rtsp_read_packet(rtsp_ctx, streamChannelId, ReadSize, rcv_buf,
	                          buf_size,data_ptr, data_size, rtp_head);

    return result;
}

#if RTSP_UDP
static int rtsp_read_rtp_over_udp(yc_rtsp_s *rtsp_ctx,
                                  char *rcv_buf,
                                  unsigned short buf_size,
                                  char **data_ptr,
                                  unsigned short *data_size,
                                  yc_rtp_packet_s **rtp_head)
{
    fd_set read_set;
    struct timeval tv;
    int ret;
    int max_fd;
    u32 tmout_cnt = 0;

redo:
    FD_ZERO(&read_set);
    FD_SET(rtsp_ctx->prvi->rtp_skt_num, &read_set);
    FD_SET(rtsp_ctx->prvi->rtcp_skt_num, &read_set);

    tv.tv_sec  = RTP_RCV_WAIT_TIME;
    tv.tv_usec = 0;

    max_fd = FFMAX(rtsp_ctx->prvi->rtp_skt_num, rtsp_ctx->prvi->rtcp_skt_num);

    ret = select(max_fd + 1, &read_set, NULL, NULL, &tv);
    if (ret > 0)
    {
        if (rtsp_ctx->prvi->is_retry)
            rtsp_ctx->prvi->is_retry = 0;

        if (FD_ISSET(rtsp_ctx->prvi->rtp_skt_num, &read_set))
        {
redo1:
            ret = recv(rtsp_ctx->prvi->rtp_skt_num, rcv_buf, buf_size, 0);
            if (ret > 0)
            {
                ret = rtp_read(rtsp_ctx, 0, rcv_buf, ret, data_ptr, data_size, rtp_head);
            }
            else if (ret == 0)
            {
                rtp_printf("rtp skt was closed\n");
                ret = RTSP_ERRCODE_SOCKET_CLOSE;
            }
            else
            {
                if ((errno == EINTR) || (errno == EWOULDBLOCK) || (errno == EAGAIN))
    			{
                    rtp_printf("error reading rtp socket, but continue...\n");
                    goto redo1;
    			}
            }

            FD_CLR(rtsp_ctx->prvi->rtp_skt_num, &read_set);
        }

        if (FD_ISSET(rtsp_ctx->prvi->rtcp_skt_num, &read_set))
        {
redo2:
            ret = recv(rtsp_ctx->prvi->rtcp_skt_num, rcv_buf, buf_size, 0);
            if (ret > 0)
            {
                ret = rtcp_read(rtsp_ctx, rcv_buf, ret, data_ptr, data_size);
            }
            else if (ret == 0)
            {
                rtp_printf("rtcp skt was closed\n");
                ret = RTSP_ERRCODE_SOCKET_CLOSE;
            }
            else
            {
                if ((errno == EINTR) || (errno == EWOULDBLOCK) || (errno == EAGAIN))
    			{
                    rtp_printf("error reading rtcp socket, but continue...\n");
                    goto redo2;
    			}
            }

            FD_CLR(rtsp_ctx->prvi->rtcp_skt_num, &read_set);
        }
    }
    else if (ret == 0)
    {
        if (rtsp_ctx->prvi->is_retry)
        {
            ret = rtsp_resetup(rtsp_ctx);
            if (0 == ret)
            {
                ret = rtsp_read(rtsp_ctx, rcv_buf, buf_size, data_ptr, data_size, rtp_head);
            }
        }
        else
        {
            tmout_cnt++;
            if (tmout_cnt > (RTP_RCV_WAIT_TIME * 10))/* stop when not recv after 10s */
                ret = RTSP_ERRCODE_RECV_TIMEOUT;
            else
                goto redo;
        }
    }

    if ((tmout_cnt > 2) && (1 == ret))/* rtcp packet in 2s, may be rtp stream end */
        ret = RTSP_ERRCODE_RECV_TIMEOUT;

    return ret;
}
#endif

/* -5 user cancle, -4 rcv valid, -3 rcv timeout, -2 socket close, -1 gernal failed(skip), 0 success(rtp), 1 rtcp(skip), 2 stream end */
int yc_rtsp_read(IN yc_rtsp_s *pRtsp, OUT char **ppData, OUT yc_rtp_packet_s **ppRtp)
{
    unsigned short data_size = 0;
	unsigned short size;
	int result = 0;
    struct rtp_cache_packet *packet;

redo:
    if (RTSP_STATE_STOP == pRtsp->prvi->state)
    {
        result = RTSP_ERRCODE_USER_CANCEL;
        goto end;
    }

    if ((0 != pRtsp->prvi->end_npt) &&
        ((pRtsp->prvi->end_npt / 1000) <= (pRtsp->prvi->curr_npt / 1000)))
    {
        rtp_printf("curr npt end\n");
        result = RTSP_ERRCODE_STREAM_END;
        goto end;
    }

    pthread_rwlock_wrlock(&pRtsp->prvi->stLock);
    packet = list_first_entry_or_null(&pRtsp->prvi->stNode, struct rtp_cache_packet, stNode);
    if (packet)
        list_del(&packet->stNode);
    pthread_rwlock_unlock(&pRtsp->prvi->stLock);

    if (packet)
    {
        if (pRtsp->prvi->block_size < packet->len)
            size = pRtsp->prvi->block_size;
        else
            size = packet->len;
        memcpy(pRtsp->prvi->buffer, packet->buf, size);
        rtsp_free(packet->buf);
        rtsp_free(packet);

        result = rtp_read(pRtsp, packet->stream_id, pRtsp->prvi->buffer, size,
                          ppData, &data_size, ppRtp);
    }
    else
    {
#if RTSP_UDP
        if (RTSP_TRANSPORT_MODE_UDP == pRtsp->prvi->tmode)
        {
            result = rtsp_read_rtp_over_udp(pRtsp,
                                            pRtsp->prvi->buffer,
                                            pRtsp->prvi->block_size,
                                            ppData, &data_size, ppRtp);
        }
        else if (RTSP_TRANSPORT_MODE_TCP == pRtsp->prvi->tmode)
#endif
        {
            result = rtsp_read_rtp_over_tcp(pRtsp,
                                            pRtsp->prvi->buffer,
                                            pRtsp->prvi->block_size,
                                            ppData, &data_size, ppRtp);
        }
    }

end:
    if (RTSP_ERRCODE_SUCCESS == result)
    {
        result = data_size;
    }
    else if (RTSP_ERRCODE_RTCP_PACKET == result)
    {
        goto redo;
    }
    else if (RTSP_ERRCODE_STREAM_END == result)
    {
        result = YC_ERROR_EOF;
    }
    else if (RTSP_ERRCODE_RECV_TIMEOUT == result)
    {
        result = YC_ERROR_TIMEOUT;
    }
    else
    {
        result = YC_ERROR_FAILED;
    }

	return result;
}

static int rtsp_cache_packet(yc_rtsp_s *rtsp_ctx, unsigned char stream_id, unsigned int size)
{
    int ret;
    char *buf;
    struct rtp_cache_packet *packet;

    buf = rtsp_malloc(size);
    if (!buf)
    {
        rtp_printf("malloc cache buf failed\r\n");
        return RTSP_ERRCODE_GENERAL_FAILED;
    }

    packet = rtsp_malloc(sizeof(struct rtp_cache_packet));
    if (!packet)
    {
        rtp_printf("malloc cache packet failed\r\n");
        rtsp_free(buf);
        return RTSP_ERRCODE_GENERAL_FAILED;
    }

    memset(buf, 0, size);
    ret = rtp_read_by_size(rtsp_ctx, buf, size, size);
    if (0 >= ret)
    {
        rtp_printf("rtp_read_cache_by_size failed\n");
        rtsp_free(packet);
        rtsp_free(buf);
        return RTSP_ERRCODE_GENERAL_FAILED;
    }

    memset(packet, 0, sizeof(struct rtp_cache_packet));
    packet->buf = buf;
    packet->len = size;
    packet->stream_id = stream_id;

    pthread_rwlock_wrlock(&rtsp_ctx->prvi->stLock);
    list_add(&packet->stNode, &rtsp_ctx->prvi->stNode);
    pthread_rwlock_unlock(&rtsp_ctx->prvi->stLock);

    return RTSP_ERRCODE_SUCCESS;
}

static void rtsp_cache_packet_destroy(yc_rtsp_s *pRtsp)
{
    struct rtp_cache_packet *packet;
    struct rtp_cache_packet *next;

    pthread_rwlock_wrlock(&pRtsp->prvi->stLock);
    list_for_each_entry_safe(packet, next, &pRtsp->prvi->stNode, stNode)
    {
        list_del(&packet->stNode);
        rtsp_free(packet->buf);
        rtsp_free(packet);
    }
    pthread_rwlock_unlock(&pRtsp->prvi->stLock);

    return;
}

static int rtsp_redir_is_space(int c)
{
    return c == ' ' || c == '\t' || c == '\n' || c == '\r';
}

static void rtsp_skip_spaces(const char **pp)
{
    const char *p;
    p = *pp;
    while (rtsp_redir_is_space(*p))
        p++;
    *pp = p;
}

static void rtsp_get_word_sep(char *buf, int buf_size, const char *sep,
                         const char **pp)
{
    const char *p;
    char *q;

    p = *pp;
    if (*p == '/')
        p++;
    rtsp_skip_spaces(&p);
    q = buf;
    while (!strchr(sep, *p) && *p != '\0') {
        if ((q - buf) < buf_size - 1)
            *q++ = *p;
        p++;
    }
    if (buf_size > 0)
        *q = '\0';
    *pp = p;
}

static void rtsp_get_word(char *buf, int buf_size, const char **pp)
{
    const char *p;
    char *q;

    p = *pp;
    rtsp_skip_spaces(&p);
    q = buf;
    while (!rtsp_redir_is_space(*p) && *p != '\0') {
        if ((q - buf) < buf_size - 1)
            *q++ = *p;
        p++;
    }
    if (buf_size > 0)
        *q = '\0';
    *pp = p;
}

static char* rtsp_str_dup(const char* str)
{
#if 0
	char* copy;
	unsigned int len;
	if (str == NULL) return NULL;
	len = strlen(str) + 1;
	copy = (char*)rtsp_malloc(len*sizeof(char));

	if (copy != NULL) {
		memcpy(copy, str, len);
	}
	return copy;
#endif
    return strdup(str);
}

static char* rtsp_str_dup_size(char* str)
{
	char* copy;
	unsigned int len;
	if (str == NULL) return NULL;
	len = strlen(str) + 1;
	copy = (char *)rtsp_malloc(len*sizeof(char));

	return copy;
}

static char* rtsp_get_line(char* startOfLine)
{
	// returns the start of the next line, or NULL if none
	char* ptr;
	for (ptr = startOfLine; *ptr != '\0'; ++ptr)
	{
		if (*ptr == '\r' || *ptr == '\n')
		{
			// We found the end of the line
			*ptr++ = '\0';
			if (*ptr == '\n') ++ptr;
			return ptr;
		}
	}

	return NULL;
}

static unsigned long long rtsp_str2ms(char *str)
{
#if 0
	char temp[32];
	int index = 0;
	int j = 0;
	char *endstr;
	int k;
	int i,num;
    int PotPos = 0;
    unsigned long long endtime = 0;
    unsigned long long fMaxPlayEndTime = 0;

	for(index = 0;index<strlen(str);index++)
	{
		if(str[index]!='.')
		{
			temp[j] = str[index];
			j++;
		}
		else
		{
			for(k=index+1;k<strlen(str);k++)
			{
				temp[j] = str[k];
				j++;
				PotPos++;
			}
			break;
		}

	}
	temp[j] = '\0';


	endtime = strtoul(temp,&endstr,10);

	if (endtime > fMaxPlayEndTime)
	{
		if(PotPos>=3)
		{
			num = 1;
			for(i = 0;i<PotPos-3;i++)
			{
				num = 10*num;
			}
			fMaxPlayEndTime = endtime/num;
		}
		else
		{
			num = 1;
			for(i = 0;i<3-PotPos;i++)
			{
				num = 10*num;
			}
			fMaxPlayEndTime = endtime*num;
		}
		PotPos = 1000;

	}

	return fMaxPlayEndTime;
#endif
    return atof(str) * 1000;
}

static char* rtsp_sdp_parse_line(char* inputLine)
{
	char *ptr;

	for (ptr = inputLine; *ptr != '\0'; ++ptr)
	{
		if (*ptr == '\r' || *ptr == '\n')
		{
			*ptr++ = '\0';
			while (*ptr == '\r' || *ptr == '\n') *ptr++ = '\0';
			if (ptr[0] == '\0') ptr = NULL; // special case for end
			break;
		}
	}

	if (inputLine[0] == '\r' || inputLine[0] == '\n') return NULL;
	if (strlen(inputLine) < 2 || inputLine[1] != '='
		    || inputLine[0] < 'a' || inputLine[0] > 'z')
	{
		rtsp_printf("Invalid SDP line \n");
		return NULL;
	}

	return ptr;
}

static char * rtsp_sdp_parse_attribute_rtpmap(yc_rtsp_s *rtsp_ctx,
                                              char* sdpLine,
                                              unsigned int *rtpTimestampFrequency,
                                              unsigned int *fnumChannels)
{
    char *ret = NULL;
	unsigned int rtpmapPayloadFormat;
	unsigned int numChannels = 0;
	char* codecName = rtsp_str_dup_size(sdpLine);

	if (sscanf(sdpLine, "a=rtpmap: %u %[^/]/%u/%u",
			&rtpmapPayloadFormat, codecName, rtpTimestampFrequency,
			&numChannels) == 4
		|| sscanf(sdpLine, "a=rtpmap: %u %[^/]/%u",
			&rtpmapPayloadFormat, codecName, rtpTimestampFrequency) == 3
		|| sscanf(sdpLine, "a=rtpmap: %u %s",
			&rtpmapPayloadFormat, codecName) == 2)
	{
        if (0 == strncmp("x-wms-fec", codecName, 9))
            goto skip;

		*fnumChannels = numChannels;
		ret = rtsp_str_dup(codecName);
		rtsp_free(codecName);
		return ret;

	}

skip:
	rtsp_free(codecName);

	return NULL;
}

static char * rtsp_sdp_parse_attribute_control(char* sdpLine)
{
    char *ret = NULL;
	char* controlPath = rtsp_str_dup_size(sdpLine); // ensures we have enough space

	if (sscanf(sdpLine, "a=control:%s", controlPath) == 1)
	{
	    ret = rtsp_str_dup(controlPath);
	    rtsp_free(controlPath);
		return ret;
	}
	rtsp_free(controlPath);

	return NULL;
}

static int rtsp_sdp_parse_attribute_range(char* sdpLine, unsigned long long *PlayTimeLen)
{
	//int parseSuccess = -1;
	char playEndTime[32];
#if 0
	char temp[32];
	int index = 0;
	int j = 0;
	unsigned long endtime = 0;
	char *endstr;
	int k;
	int i,num;
    int PotPos = 0;
#endif
    unsigned long long fMaxPlayBeginTime = 0;
    unsigned long long fMaxPlayEndTime = 0;

    sdpLine += 12;
    rtsp_get_word_sep(playEndTime, 32, "-", (const char **)&sdpLine);
    fMaxPlayBeginTime = rtsp_str2ms(playEndTime);
    sdpLine++;
    rtsp_get_word(playEndTime, 32, (const char **)&sdpLine);
    fMaxPlayEndTime = rtsp_str2ms(playEndTime);

#if 0
	if (sscanf(sdpLine, "a=range:npt=0-  %s", playEndTime)==1)
	{
		parseSuccess = 0;

		for(index = 0;index<strlen(playEndTime);index++)
		{
			if(playEndTime[index]!='.')
			{
				temp[j] = playEndTime[index];
				j++;
			}
			else
			{
				for(k=index+1;k<strlen(playEndTime);k++)
				{
					temp[j] = playEndTime[k];
					j++;
					PotPos++;
				}
				break;
			}

		}
		temp[j] = '\0';


		endtime = strtoul(temp,&endstr,10);

		if (endtime > fMaxPlayEndTime)
		{
			if(PotPos>=3)
			{
				num = 1;
				for(i = 0;i<PotPos-3;i++)
				{
					num = 10*num;
				}
				fMaxPlayEndTime = endtime/num;
			}
			else
			{
				num = 1;
				for(i = 0;i<3-PotPos;i++)
				{
					num = 10*num;
				}
				fMaxPlayEndTime = endtime*num;
			}
			PotPos = 1000;

		}

        fMaxPlayEndTime = rtsp_str2ms(playEndTime);
	    rtsp_printf("fMaxPlayEndTime is %llu\n",fMaxPlayEndTime);
	}
#endif

    fMaxPlayEndTime = fMaxPlayEndTime - fMaxPlayBeginTime;
	rtsp_printf("PlayTimeLen is %llu\n",fMaxPlayEndTime);

	*PlayTimeLen = fMaxPlayEndTime;

	return 0;//parseSuccess;
}

static char *rtsp_sdp_parse_attribute_title(char* sdpLine)
{
    char *begin, *end;
    char *title;
    unsigned int len = 0;

    begin = sdpLine + 2;
    end = begin;

    while ((*end != '\r') && (*end != '\n'))
    {
        end++;
    }

    len = end - begin;
    title  = rtsp_malloc(len + 1);
    if (NULL != title)
    {
        memcpy(title, begin, len);
        title[len] = '\0';
    }

    return title;
}

static char *rtsp_sdp_parse_attribute_tool(char* sdpLine)
{
    char *begin, *end;
    char *tool;
    unsigned int len = 0;

    begin = sdpLine + 7;/* "a=tool:" */
    end = begin;

    while ((*end != '\r') && (*end != '\n'))
    {
        end++;
    }

    len = end - begin;
    tool  = rtsp_malloc(len + 1);
    if (NULL != tool)
    {
        memcpy(tool, begin, len);
        tool[len] = '\0';
        rtsp_printf("server name: '%s'\r\n", tool);
    }

    return tool;
}

#if RTSP_ASF
#define RTSP_ARRAY_ELEMS(a) (sizeof(a) / sizeof((a)[0]))

#define RTSP_WL32(p, d) do { \
                    ((unsigned char*)(p))[0] = (d); \
                    ((unsigned char*)(p))[1] = (d)>>8; \
                    ((unsigned char*)(p))[2] = (d)>>16; \
                    ((unsigned char*)(p))[3] = (d)>>24; } while(0)

#define RTSP_RL32(x) ((((const unsigned char*)(x))[3] << 24) | \
                    (((const unsigned char*)(x))[2] << 16) | \
                    (((const unsigned char*)(x))[1] <<  8) | \
                     ((const unsigned char*)(x))[0])

#define RTSP_RL64(x)  (((uint64_t)((const unsigned char*)(x))[7] << 56) | \
                     ((uint64_t)((const unsigned char*)(x))[6] << 48) | \
                     ((uint64_t)((const unsigned char*)(x))[5] << 40) | \
                     ((uint64_t)((const unsigned char*)(x))[4] << 32) | \
                     ((uint64_t)((const unsigned char*)(x))[3] << 24) | \
                     ((uint64_t)((const unsigned char*)(x))[2] << 16) | \
                     ((uint64_t)((const unsigned char*)(x))[1] <<  8) | \
                      (uint64_t)((const unsigned char*)(x))[0])

typedef unsigned char rtsp_asf_guid[16];

static rtsp_asf_guid rtsp_asf_header = {
    0x30, 0x26, 0xB2, 0x75, 0x8E, 0x66, 0xCF, 0x11, 0xA6, 0xD9, 0x00, 0xAA, 0x00, 0x62, 0xCE, 0x6C
};

static rtsp_asf_guid rtsp_asf_file_header = {
    0xA1, 0xDC, 0xAB, 0x8C, 0x47, 0xA9, 0xCF, 0x11, 0x8E, 0xE4, 0x00, 0xC0, 0x0C, 0x20, 0x53, 0x65
};

/**
 * From MSDN 2.2.1.4, we learn that ASF data packets over RTP should not
 * contain any padding. Unfortunately, the header min/max_pktsize are not
 * updated (thus making min_pktsize invalid). Here, we "fix" these faulty
 * min_pktsize values in the ASF file header.
 * @return 0 on success, <0 on failure (currently -1).
 */
static int rtp_asf_fix_header(unsigned char *buf, int len)
{
    unsigned char *p = buf, *end = buf + len;

    if (len < sizeof(rtsp_asf_guid) * 2 + 22 ||
        memcmp(p, rtsp_asf_header, sizeof(rtsp_asf_guid))) {
        return -1;
    }
    p += sizeof(rtsp_asf_guid) + 14;
    do {
        unsigned long long chunksize = RTSP_RL64(p + sizeof(rtsp_asf_guid));
        int skip = 6 * 8 + 3 * 4 + sizeof(rtsp_asf_guid) * 2;
        if (memcmp(p, rtsp_asf_file_header, sizeof(rtsp_asf_guid))) {
            if (chunksize > end - p)
                return -1;
            p += chunksize;
            continue;
        }

        if (end - p < 8 + skip)
            break;
        /* skip most of the file header, to min_pktsize */
        p += skip;
        if (RTSP_RL32(p) == RTSP_RL32(p + 4)) {
            /* and set that to zero */
            /* pad 0 may be error */
            //RTSP_WL32(p, 0);
            return 0;
        }
        break;
    } while (end - p >= sizeof(rtsp_asf_guid) + 8);

    return -1;
}

static const char base64_pad = '=';

static const short base64_reverse_table[256] = {
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -1, -1, -2, -2, -1, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-1, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, 62, -2, -2, -2, 63,
	52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -2, -2, -2, -2, -2, -2,
	-2,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
	15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -2, -2, -2, -2, -2,
	-2, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
	41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
	-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2
};

static int base64_decode(const unsigned char *str, int length, unsigned char *result) /* {{{ */
{
	const unsigned char *current = str;
	int ch, i = 0, j = 0, k;
	/* this sucks for threaded environments */

	/* run through the whole string, converting as we go */
	while ((ch = *current++) != '\0' && length-- > 0) {
		if (ch == base64_pad) {
			if (*current != '=' && ((i % 4) == 1)) {
				if ((i % 4) != 1) {
					while (isspace(*(++current))) {
						continue;
					}
					if (*current == '\0') {
						continue;
					}
				}

				return 0;
			}
			continue;
		}

		ch = base64_reverse_table[ch];
		if (ch == -1) { /* a space or some other separator character, we simply skip over */
			continue;
		} else if (ch == -2) {
			return 0;
		}

		switch(i % 4) {
		case 0:
			result[j] = ch << 2;
			break;
		case 1:
			result[j++] |= ch >> 4;
			result[j] = (ch & 0x0f) << 4;
			break;
		case 2:
			result[j++] |= ch >>2;
			result[j] = (ch & 0x03) << 6;
			break;
		case 3:
			result[j++] |= ch;
			break;
		}
		i++;
	}

	k = j;
	/* mop things up if we ended on a boundary */
	if (ch == base64_pad) {
		switch(i % 4) {
		case 1:
			return 0;
		case 2:
			k++;
		case 3:
			result[k] = 0;
		}
	}

	//result[j] = '\0';

	return j;
}

static void rtsp_sdp_parse_attribute_wms(yc_rtsp_s *rtsp_ctx, char* sdpLine)
{
    char *p = sdpLine + 53;
    int len = strlen(p);// * 6 / 8;
    unsigned char *buf;

    buf = rtsp_malloc(len);
    if (!buf)
        return;

    len = base64_decode((unsigned char *)p, len, buf);

    if (rtp_asf_fix_header(buf, len) < 0)
    {
        rtsp_printf("Failed to fix invalid RTSP-MS/ASF min_pktsize\n");
        rtsp_free(buf);
    }
    else
    {
        rtsp_ctx->prvi->asf_head = buf;
        rtsp_ctx->prvi->asf_len  = len;
        rtsp_printf("asf head len = '%d'.\r\n", len);
    }

    return;
}

static void rtsp_sdp_parse_attribute_maxps(yc_rtsp_s *rtsp_ctx, char* sdpLine)
{
    char *pos = sdpLine + 8;

    rtsp_ctx->prvi->maxps = atoi(pos);
    rtsp_printf("asf maxps '%hu'.\r\n", rtsp_ctx->prvi->maxps);

    return;
}
#endif

static int rtsp_sdp_init(yc_rtsp_s *rtsp_ctx, char* sdpDescription)
{
	int Num = 0;
	char*  sdpLine = sdpDescription;
	char* nextSDPLine = NULL;
	yc_rtsp_media_s *fSubsessionsHead;
	yc_rtsp_media_s *fSubsessionsTail;
	yc_rtsp_media_s* subsession;
	char* mediumName;
	char * CodecName;
	char * ControlPath = NULL;
	unsigned int payloadFormat;
    char *title = NULL;
    char *tool = NULL;
    unsigned long long fMaxPlayEndTime = 0;
    unsigned char found = 0;

	if (sdpDescription == NULL) return -1;
	fSubsessionsHead = fSubsessionsTail = NULL;
	while (1)
	{
		if ((nextSDPLine = rtsp_sdp_parse_line(sdpLine)) == NULL)
		{
		    if (title)
		        rtsp_free(title);
			return -1;
		}
		if (sdpLine[0] == 's')
		{
            title = rtsp_sdp_parse_attribute_title(sdpLine);
		}
#if RTSP_ASF
		else if (0 == strncmp(sdpLine, "a=pgmpu:data:application/vnd.ms.wms-hdr.asfv1;base64,", 53))
		{
		    if (0 == strncmp(rtsp_ctx->prvi->server, "WMServer/", 9))
                rtsp_sdp_parse_attribute_wms(rtsp_ctx, sdpLine);
		}
		else if (0 == strncmp(sdpLine, "a=maxps:", 8))
		{
		    rtsp_sdp_parse_attribute_maxps(rtsp_ctx, sdpLine);
		}
#endif
		else if (0 == strncmp(sdpLine, "a=tool:", 7))
		{
		    if (NULL == rtsp_ctx->prvi->server)
                tool = rtsp_sdp_parse_attribute_tool(sdpLine);
		}
		else if (0 == strncmp(sdpLine, "a=range:", 8))
		{
            rtsp_sdp_parse_attribute_range(sdpLine, &fMaxPlayEndTime);
 		}
		//else if (sdpLine[0] == 'm')
		else if (0 == strncmp(sdpLine, "m=audio", 7))
		{
			Num++;
			found = 1;
			break;
		}
		else if (0 == strncmp(sdpLine, "m=vedio", 7))
		{
			Num++;
			found = 1;
			break;
		}
		sdpLine = nextSDPLine;
		//if (!parseSDPAttribute_range(sdpLine)) continue;//check a=range:npt=
	}

	while (sdpLine != NULL)
	{
		subsession = rtsp_malloc(sizeof(yc_rtsp_media_s));
		if (subsession == NULL)
		{
			rtsp_printf("Unable to create new rtsp_media_info\n");
			if (title)
		        rtsp_free(title);
		    if (tool)
		        rtsp_free(tool);
			return -1;
		}

        memset(subsession, 0, sizeof(yc_rtsp_media_s));
        if (NULL == rtsp_ctx->prvi->title)
            rtsp_ctx->prvi->title = title;
        if (NULL == rtsp_ctx->prvi->server)
            rtsp_ctx->prvi->server = tool;
        subsession->time_len = fMaxPlayEndTime;

		if (fSubsessionsTail == NULL)
		{
			fSubsessionsHead = fSubsessionsTail = subsession;
		}
		else
		{
			fSubsessionsTail->next = subsession;
			subsession->next = NULL;
			fSubsessionsTail = subsession;
		}

		mediumName = rtsp_str_dup_size(sdpLine);

		if (sscanf(sdpLine, "m=%s %hu RTP/AVP %u",mediumName, &subsession->client_port, &payloadFormat) != 3
			|| payloadFormat > 127)
		{
			rtsp_printf("Bad SDP line\n");
			rtsp_free(mediumName);
			rtsp_free(subsession);
			if (title)
		        rtsp_free(title);
		    if (tool)
		        rtsp_free(tool);
			return -1;
		}

		subsession->media_name = rtsp_str_dup(mediumName);
		rtsp_free(mediumName);
		subsession->payload_type = payloadFormat;

		while (1)
		{
			sdpLine = nextSDPLine;
			if (sdpLine == NULL) break; // we've reached the end
			nextSDPLine = rtsp_sdp_parse_line(sdpLine);

			if ((sdpLine[0] == 'm') &&
			    (0 == strncmp(sdpLine, "m=audio", 7)))
			{
				Num++;
				found = 1;
				break; // we've reached the next subsession
			}
			if ((sdpLine[0] == 'm') &&
			    (0 == strncmp(sdpLine, "m=vedio", 7)))
			{
				Num++;
				found = 1;
				break; // we've reached the next subsession
			}
			else if (sdpLine[0] == 'm')
			{
			    found = 0;
                continue;/* skip other streamex. tsapplication... */
			}
            else if (0 == strncmp(sdpLine, "a=rtpmap", 8) && found)
            {
                if ((CodecName = rtsp_sdp_parse_attribute_rtpmap(rtsp_ctx, sdpLine, &(subsession->timestamp_frequency), &(subsession->channel_num))) != NULL)
    			{
    				subsession->codec_name = rtsp_str_dup(CodecName);
    				rtsp_free(CodecName);
    			}
            }
            else if (0 == strncmp(sdpLine, "a=control", 9) && found)
            {
                if ((ControlPath = rtsp_sdp_parse_attribute_control(sdpLine))!=NULL)
    			{
    			    if (subsession->control_path)
    			    {
    			        rtsp_free(subsession->control_path);
    			        subsession->control_path = NULL;
    			    }
    				subsession->control_path = rtsp_str_dup(ControlPath);
    				rtsp_free(ControlPath);
    			}
            }

			if (nextSDPLine == NULL)
			{
				rtsp_ctx->prvi->session_num = Num;
				rtsp_ctx->media = fSubsessionsHead;
				return 0;
			}
		}
	}

	rtsp_ctx->prvi->session_num = Num;
	rtsp_ctx->media = fSubsessionsHead;

	return 0;
}

static void rtsp_clear_up(yc_rtsp_s *pRtsp)
{
    yc_rtsp_media_s *curr = NULL;

    if (NULL != pRtsp)
    {
        if (pRtsp->prvi != NULL)
        {
            if (YC_TIMER_ID_INVALID != pRtsp->prvi->timer)
                yc_timer_del(pRtsp->prvi->timer);

            if (NULL != pRtsp->prvi->buffer)
                rtsp_free(pRtsp->prvi->buffer);

            rtsp_cache_packet_destroy(pRtsp);
            pthread_rwlock_destroy(&pRtsp->prvi->stLock);

        	if(pRtsp->prvi->url != NULL)
        	    rtsp_free(pRtsp->prvi->url);
        	if(pRtsp->prvi->session_id != NULL)
        	    rtsp_free(pRtsp->prvi->session_id);
        	if (pRtsp->prvi->server)
    	        rtsp_free(pRtsp->prvi->server);
    	    if (pRtsp->prvi->title)
    	        rtsp_free(pRtsp->prvi->title);
    	    if (pRtsp->prvi->asf_head)
                rtsp_free(pRtsp->prvi->asf_head);
        	if(pRtsp->prvi->skt_num>=0)
        	{
        	    rtsp_printf("##clsoe socketNum is %d\n", pRtsp->prvi->skt_num);
                close(pRtsp->prvi->skt_num);
            }
            if(pRtsp->prvi->rtp_skt_num>=0)
                close(pRtsp->prvi->rtp_skt_num);
            if(pRtsp->prvi->rtcp_skt_num>=0)
                close(pRtsp->prvi->rtcp_skt_num);

            rtsp_free(pRtsp->prvi);
        }

        while (pRtsp->media != NULL)
        {
            if (pRtsp->media->media_name)
                rtsp_free(pRtsp->media->media_name);
            if (pRtsp->media->codec_name)
                rtsp_free(pRtsp->media->codec_name);
            if (pRtsp->media->control_path)
                rtsp_free(pRtsp->media->control_path);

            curr = pRtsp->media;
            pRtsp->media = pRtsp->media->next;
            rtsp_free(curr);
        }

        rtsp_free(pRtsp);
    }

	return;
}

static int rtsp_parse_response_code(char* line, unsigned int * responseCode)
{
	if (sscanf(line, "%*s%u", responseCode) != 1)
	{
		rtsp_printf("no response code in line\n");
		return -1;
	}

	return 0;
}

#if 0
static int rtsp_block_until_writeable(int socket)//, struct timeval* timeout)
{
	int result = -1;
	unsigned int numFds;
	fd_set wd_set;
	do
	{
		FD_ZERO(&wd_set);
		if (socket < 0) break;
		FD_SET((unsigned) socket, &wd_set);
		numFds = socket+1;

		result = select(numFds, NULL, &wd_set, NULL,0);// timeout);
		if (result == 0)
		{
			break; // this is OK - timeout occurred
		}
		else if(result <= 0)
		{
			rtsp_printf( "select() error\n ");
			break;
		}

		if (!FD_ISSET(socket, &wd_set))
		{
			rtsp_printf( "select() error - !FD_ISSET\n");
			break;
		}
	} while (0);
	return result;
};
#endif

static int rtsp_block_until_readable(int socket)//, struct timeval* timeout)
{
	fd_set rd_set;
	int result = -1;
	unsigned int numFds;

	do
	{
		FD_ZERO(&rd_set);
		if (socket < 0) break;
		FD_SET((unsigned) socket, &rd_set);
		numFds = socket+1;
		result = select(numFds, &rd_set, NULL, NULL,0);
		if (result == 0)
		{
			break; // this is OK - timeout occurred
		}
		else if (result <= 0)
		{
			rtsp_printf( "select() error\n ");
			break;
		}

		if (!FD_ISSET(socket, &rd_set))
		{
			rtsp_printf( "select() error - !FD_ISSET\n");
			break;
		}
	} while (0);

	return result;
}

static int rtsp_skip_packet(yc_rtsp_s *rtsp_ctx)
{
    int i;
    int ret = 0;
    int result;
    unsigned char streamChannelId;
	unsigned short size;

redo1:
	result = recv(rtsp_ctx->prvi->skt_num, &streamChannelId, 1, 0);
	if (result == 0)
	{
	    rtsp_printf("skip_packet sokcet was closed.\n");
        return -1;
	}
	else if (result < 0)
	{
        if (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN)
        {
            rtsp_printf("skip_packet sokcet error, continue...\n");
            goto redo1;
        }
        else
        {
            rtsp_printf("skip_packet sokcet error.\n");
            return -1;
        }
	}

redo2:
    result = recv(rtsp_ctx->prvi->skt_num, (unsigned char*)&size, 2, 0);
    if (result == 0)
	{
	    rtsp_printf("skip_packet sokcet was closed2.\n");
        return -1;
	}
	else if (result < 0)
	{
        if (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN)
        {
            rtsp_printf("skip_packet sokcet error2, continue...\n");
            goto redo2;
        }
        else
        {
            rtsp_printf("skip_packet sokcet error2.\n");
            return -1;
        }
	}

    size = ntohs(size);

    result = 0;
    for (i = 0; i < RTSP_CHANNEL_NUM_MAX; i++)
    {
        if (streamChannelId == rtsp_ctx->prvi->rtp_channel[i])
        {
            result = 1;
            ret = rtsp_cache_packet(rtsp_ctx, streamChannelId, size);
            break;
        }
    }

    if (!result)
    {
        while (size--)
        {
redo3:
            result = recv(rtsp_ctx->prvi->skt_num, &streamChannelId, 1, 0);
            if (result == 0)
        	{
        	    rtsp_printf("skip_packet sokcet was closed3.\n");
                return -1;
        	}
        	else if (result < 0)
        	{
                if (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN)
                {
                    rtsp_printf("skip_packet sokcet error3, continue...\n");
                    goto redo3;
                }
                else
                {
                    rtsp_printf("skip_packet sokcet error3.\n");
                    return -1;
                }
        	}
        }
    }

    return ret;
}

static void rtsp_parse_line(yc_rtsp_s *rtsp_ctx, char *buf, unsigned short len)
{
    char *pos;
    char temp[128];
    char SessionId[64];

    if (0 == strncmp(buf, "Server: ", 8))
    {
        pos = buf + 8;
        while (pos[0] != '\r' && pos[0] != '\n')
        {
            pos++;
        }

        if (rtsp_ctx->prvi->server)
            return;

        rtsp_ctx->prvi->server = rtsp_malloc(pos - (buf + 8) + 1);
        if (rtsp_ctx->prvi->server)
        {
            memcpy(rtsp_ctx->prvi->server, buf + 8, pos - (buf + 8));
            rtsp_ctx->prvi->server[pos - (buf + 8)] = '\0';
            rtsp_printf("server name: '%s'\r\n", rtsp_ctx->prvi->server);
        }
    }
    else if (0 == strncmp(buf, "Range: npt=", 11))
	{
        pos = buf + 11;
	    rtsp_get_word_sep(temp, 128, "-", (const char **)&pos);
	    rtsp_ctx->prvi->begin_npt = rtsp_str2ms(temp);
	    pos++;
	    if ('\0' == *pos || '\r' == *pos || '\n' == *pos)
	    {
            rtsp_ctx->prvi->end_npt = 0;
	    }
	    else
	    {
    	    rtsp_get_word(temp, 128, (const char **)&pos);
            rtsp_ctx->prvi->end_npt = rtsp_str2ms(temp);
        }
        rtsp_printf("start npt: '%llu', end npt: '%llu'\r\n", rtsp_ctx->prvi->begin_npt, rtsp_ctx->prvi->end_npt);
	}
	else if (0 == strncmp(buf, "Scale: ", 7))
	{
        pos = buf + 7;
	    rtsp_ctx->prvi->scale = rtsp_str2ms(pos);
        rtsp_printf("play scale: '%hu'\r\n", rtsp_ctx->prvi->scale);
	}
	else if (0 == strncmp(buf, "RTP-Info:", 9))
	{
        pos = buf + 9;
        do
        {
            if (0 == strncmp(pos, "seq=", 4))
            {
                pos = pos + 4;
                rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
                rtsp_ctx->prvi->start_seq = atoi(temp);
                rtsp_ctx->prvi->is_first_packet = 1;
                rtsp_printf("rtp begin seqno: '%hu'\r\n", rtsp_ctx->prvi->start_seq);
                //break;
                if ('\0' == pos[0])
                    break;
                pos++;
            }
            else if (0 == strncmp(pos, "rtptime=", 8))
            {
                pos = pos + 8;
                rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
                rtsp_ctx->prvi->rtptime = atoll(temp);
                rtsp_printf("rtptime: '%u'\r\n", rtsp_ctx->prvi->rtptime);
                //break;
                if ('\0' == pos[0])
                    break;
                pos++;
            }
            else
            {
                rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
                if ('\0' == pos[0])
                    break;
                pos++;
            }
        } while(1);
	}
    else if (0 == strncmp(buf, "Transport: ", 11))
    {
        pos = buf + 11;
        if (0 == strncmp(pos, "RTP/AVP/TCP", 11))
        {
            rtsp_ctx->prvi->tmode = RTSP_TRANSPORT_MODE_TCP;
        }
#if RTSP_UDP
        else
        {
            rtsp_ctx->prvi->tmode = RTSP_TRANSPORT_MODE_UDP;
            rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
            pos++;
            rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
            if (0 == strncmp(temp, "multicast", 9))
                rtsp_ctx->prvi->tmode = RTSP_TRANSPORT_MODE_UDP_MULTICAST;
        }
#endif
        rtsp_printf("rtsp transport mode %u\r\n", rtsp_ctx->prvi->tmode);

        pos = buf;
        do
        {
            rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
            pos++;
            if (0 == strncmp(pos, "interleaved=", 12))
            {
                pos = pos + 12;
                rtsp_ctx->prvi->rtp_channel[rtsp_ctx->prvi->stream_index]  = (int)(pos[0] - '0');
                if ('-' == pos[1])
                    rtsp_ctx->prvi->rtcp_channel[rtsp_ctx->prvi->stream_index] = (int)(pos[2] - '0');
                rtsp_printf("rtp_channel[%d]='%hhu', rtcp_channel[%d]='%hhu'\n",
                            rtsp_ctx->prvi->stream_index, rtsp_ctx->prvi->rtp_channel[rtsp_ctx->prvi->stream_index],
                            rtsp_ctx->prvi->stream_index, rtsp_ctx->prvi->rtcp_channel[rtsp_ctx->prvi->stream_index]);
                rtsp_ctx->prvi->stream_index++;
                break;
            }
#if RTSP_UDP
            else if (0 == strncmp(pos, "server_port=", 12))
            {
                pos = pos + 12;
                rtsp_get_word_sep(temp, 128, "-", (const char **)&pos);
                rtsp_ctx->prvi->rtp_server_port = atoi(temp);
                pos++;
                rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
                rtsp_ctx->prvi->rtcp_server_port = atoi(temp);

                rtsp_printf("rtp_server_port='%hu', rtcp_server_port='%hu'\n",
                            rtsp_ctx->prvi->rtp_server_port,
                            rtsp_ctx->prvi->rtcp_server_port);
                break;
            }
#endif
        } while (pos[0] != '\r' && pos[0] != '\n');
    }
    else if (0 == strncmp(buf, "Public: ", 8))
	{
        pos = buf + 8;
        do
        {
            while (' ' == pos[0]) pos++;
            if (0 == strncmp(pos, "GET_PARAMETER", 13))
            {
                rtsp_ctx->prvi->server_method |= YC_BIT(RTSP_METHOD_GET_PARAMETER);
                rtsp_printf("server support GET_PARAMETER\n");
            }
            else if (0 == strncmp(pos, "PAUSE", 5))
            {
                rtsp_ctx->prvi->server_method |= YC_BIT(RTSP_METHOD_PAUSE);
                rtsp_printf("server support PAUSE\n");
            }
            /* ෽ݲע */
            rtsp_get_word_sep(temp, 128, ",", (const char **)&pos);
            pos++;
        } while (temp[0] != '\0');
	}
	else if (sscanf(buf, "Session: %s",SessionId) == 1)
	{
	    pos = SessionId;
	    rtsp_get_word_sep(temp, 128, ";", (const char **)&pos);
		if(rtsp_ctx->prvi->session_id != NULL) rtsp_free(rtsp_ctx->prvi->session_id);
		rtsp_ctx->prvi->session_id = rtsp_str_dup(temp);
		pos++;
		if (0 == strncmp(pos, "timeout=", 8))
		{
			rtsp_get_word_sep(temp, 128, "=", (const char **)&pos);/* rtsp_skip timeout */
			pos++;
			rtsp_get_word(temp, 128, (const char **)&pos);
			rtsp_ctx->prvi->timeout = atoi(temp);
            rtsp_printf("client keep-alive timeout '%hu'\n", rtsp_ctx->prvi->timeout);
        }
	}
	else if (sscanf(buf, "Blocksize: %s",temp) == 1)
	{
		rtsp_ctx->prvi->block_size = atoi(temp);
		rtsp_printf("rtp packet blocksize '%u'\n", rtsp_ctx->prvi->block_size);
	}
    else if (sscanf(buf, "Content-Length: %hu", &rtsp_ctx->prvi->sdp_len) == 1 ||
			 sscanf(buf, "Content-length: %hu", &rtsp_ctx->prvi->sdp_len) == 1)
	{
        rtsp_printf("sdp content len '%hu'\n", rtsp_ctx->prvi->sdp_len);
	}

    return;
}

static int rtsp_get_response(yc_rtsp_s *rtsp_ctx,char* presponseBuffer,unsigned int responseBufferSize)
{
    int line_count, total_len = 0;
    int len;
    char *buf, *q;
    unsigned char ch;
    char set_pam = 0;
    unsigned int seq;
    int ret;

    buf = rtsp_malloc(RTSP_READ_BUF_SIZE_MAX);
    if (!buf)
    {
        rtsp_printf("malloc recv buf failed.\n");
        return 0;
    }

    if(rtsp_block_until_readable(rtsp_ctx->prvi->skt_num) <= 0)
	{
		rtsp_printf("socket is unreadable\n");
		return 0;
	}

redo:
    line_count = 0;
    memset(presponseBuffer, 0, responseBufferSize);
    for(;;) {
        q = buf;
        for(;;) {
redo1:
            ret = recv(rtsp_ctx->prvi->skt_num, &ch, 1, 0);
            if (ret == 0)
        	{
        	    rtsp_printf("get response sokcet was closed.\n");
                break;
        	}
        	else if (ret < 0)
        	{
                if (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN)
                {
                    rtsp_printf("get response sokcet error, continue...\n");
                    goto redo1;
                }
                else
                {
                    rtsp_printf("get response sokcet error.\n");
                    break;
                }
        	}

            if (ch == '\n')
                break;
            if (ch == '$') {
                rtsp_skip_packet(rtsp_ctx);/*rtp*/
            } else if (ch != '\r') {
                if ((q - buf) < RTSP_READ_BUF_SIZE_MAX - 1)
                    *q++ = ch;
            }
        }
        *q = '\0';

        //rtsp_printf("line = '%s'\r\n", buf);

        /* test if last line */
        if (buf[0] == '\0')
        {
            if (set_pam && (set_pam < 5))/* SET_PARAMETER...retry max 5 */
            {
                set_pam++;
                goto redo;
            }

            *(presponseBuffer + total_len) = '\0';
            break;
        }

        if (line_count == 0) {
            if (0 != strncmp(buf, "RTSP/", 5))/* strcmp bug */
            {
                if (0 == strncmp(buf, "SET_PARAMETER", 13))
                    set_pam = 1;

                goto redo;
            }
            else
            {
                if (set_pam)
                    set_pam = 0;

                *q++ = '\r';
                *q   = '\n';
                total_len  = q - buf + 1;
                memcpy(presponseBuffer, buf, total_len);
            }
        }
        else
        {
            if (0 == strncmp(buf, "CSeq: ", 6))
            {
                seq = atoi(buf + 6);
                if (rtsp_ctx->prvi->cseq > seq)/* SET_PARAMETER not need_reply */
                {
                    set_pam = 1;
                    goto redo;
                }
                else
                {
                    set_pam = 0;
                }
            }
            *q++ = '\r';
            *q   = '\n';
            len = q - buf + 1;
            memcpy(presponseBuffer + total_len, buf, len);
            total_len += len;
            rtsp_parse_line(rtsp_ctx, buf, len);
        }
        line_count++;
    }

    rtsp_free(buf);

	return total_len;
}

static int rtsp_parse_url(char* url,char * address,int* portNum)
{

	char *prefix = "rtsp://";
	unsigned int prefixLength = 7;
	char* from = NULL ;
	char* to = NULL;
	unsigned int i;
	char nextChar;
	if (strncmp(url, prefix, prefixLength) != 0)
	{
		rtsp_printf("URL is not of the form\n");
		return -1;
	}

	from = &url[prefixLength];
	to = &address[0];
	for (i = 0; i < RTSP_ADDRESS_LEN; ++i)
	{
		if (*from == '\0' || *from == ':' || *from == '/')
		{
	// We've completed parsing the address
			*to = '\0';
			break;
		}
		*to++ = *from++;
	}
	if (i == RTSP_ADDRESS_LEN)
	{
		rtsp_printf("URL is too long\n");
		return -1;
	}

 	*portNum = 554; // default value
	nextChar = *from;
	if (nextChar == ':')
	{
		int portNumInt;
		if (sscanf(++from, "%d", &portNumInt) != 1)
		{
			rtsp_printf("No port number follows :%d\n",portNumInt);
			return -1;
		}
		if (portNumInt < 1 || portNumInt > 65535)
		{
			rtsp_printf("Bad port number\n");
			return -1;
		}
		*portNum = portNumInt;
	}
	//rtsp_printf("address is %s;portNum is %d \n",address,*portNum);
	return 0;
}

static int rtsp_get_addr_by_url(char* url, struct hostent **pphp, int *dest_port)
{
    struct hostent *hp;
    int destPortNum;
    char *address = NULL;

    address = rtsp_malloc(RTSP_ADDRESS_LEN);
	if (!address)
	    return -1;

	memset(address, 0, RTSP_ADDRESS_LEN);

	if (rtsp_parse_url(url, address, &destPortNum))
	    goto fail;

    hp = gethostbyname(address);
	if (hp == NULL )
	{
		rtsp_printf("Client: Cannot resolve address [%s]\n", address);
		goto fail;
	}

    rtsp_free(address);
    if (dest_port)
        *dest_port = destPortNum;
    if (pphp)
    *pphp = hp;
    return 0;
fail:
    rtsp_free(address);
    return -1;
}

#if RTSP_UDP
static int rtsp_crate_rtp_connect(yc_rtsp_s *rtsp_ctx)
{
    int ret = -1;
    struct sockaddr_in addr;
    int rtp_skt;

	rtp_skt = socket(AF_INET, SOCK_DGRAM, 0);
	if (rtp_skt < 0)
	{
		rtsp_printf("rtp: Error Opening socket\n");
		return -1;
	}

    memset(&addr, 0, sizeof(struct sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_port = htons((unsigned short)rtsp_ctx->prvi->rtp_local_port);
	addr.sin_addr.s_addr = htonl(IPADDR_ANY);

    ret = bind(rtp_skt, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));
    if (ret < 0)
	{
		rtsp_printf("rtp: Error bind socket\n");
		close(rtp_skt);
		return -1;
	}

    rtsp_ctx->prvi->rtp_skt_num = rtp_skt;
    rtsp_printf("crate rtp skt: %d\n", rtp_skt);

	return 0;
}

static int rtsp_crate_rtcp_connect(yc_rtsp_s *rtsp_ctx)
{
    struct hostent *hp;
    int ret = -1;
    struct sockaddr_in addr;
    int rtcp_skt;

	ret = rtsp_get_addr_by_url(rtsp_ctx->prvi->url, &hp, NULL);
    if (ret < 0)
    {
		rtsp_printf("get addr failed\n");
		return -1;
	}

	rtcp_skt = socket(AF_INET, SOCK_DGRAM, 0);
	if (rtcp_skt < 0)
	{
		rtsp_printf("rctp: Error Opening socket\n");
		return -1;
	}

    memset(&addr, 0, sizeof(struct sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_port = htons((unsigned short)rtsp_ctx->prvi->rtp_local_port + 1);
	addr.sin_addr.s_addr = htonl(IPADDR_ANY);

    ret = bind(rtcp_skt, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));
    if (ret < 0)
	{
		rtsp_printf("rtcp: Error bind socket\n");
		closesocket(rtcp_skt);
		return -1;
	}

    memset(&addr, 0, sizeof(struct sockaddr_in));
    memcpy(&(addr.sin_addr), hp->h_addr, hp->h_length);
	addr.sin_family = AF_INET;
	addr.sin_port = htons(rtsp_ctx->prvi->rtcp_server_port);

	ret = connect(rtcp_skt, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));
	if (ret < 0)
	{
		rtsp_printf("rtcp: Error connect socket\n");
		closesocket(rtcp_skt);
		return -1;
	}

    rtsp_ctx->prvi->rtcp_skt_num = rtcp_skt;
    rtsp_printf("crate rtcp skt: %d\n", rtcp_skt);

	return 0;
}
#endif

static int rtsp_open_connection(char* url)
{
	struct hostent *hp;
	int destPortNum;
	struct sockaddr_in server;
	int ret = -1;
	//int rcv_tm = 5000;/* ms */

	if (url == NULL)
	    return -1;

    ret = rtsp_get_addr_by_url(url, &hp, &destPortNum);
    if (ret < 0)
    {
		rtsp_printf("get addr and port failed\n");
		return -1;
	}

	ret = socket(AF_INET, SOCK_STREAM, 0);
	if (ret < 0)
	{
		rtsp_printf("Client: Error Opening socket\n");
		return -1;
	}

	memset(&server,0,sizeof(struct sockaddr_in));
	memcpy(&(server.sin_addr),hp->h_addr,hp->h_length);
	server.sin_family = AF_INET;
	server.sin_port = htons((unsigned short)destPortNum);
	if (connect(ret, (struct sockaddr*)&server, sizeof(struct sockaddr_in))!= 0)
	{
		rtsp_printf("connect() failed\n");
		close(ret);
		ret = -1;
	}
    else
    {
        rtsp_printf("connect() successful \n");

#if 0
        if (setsockopt(ret, SOL_SOCKET, SO_RCVTIMEO, (void*)&rcv_tm, sizeof(rcv_tm)))
            printf("set rcv timeout failed.\r\n");
#endif
    }

	return ret;
}

static int rtsp_send_quest(yc_rtsp_s *rtsp_ctx, char *cmd)
{
    int ret;

#if 0
    if (rtsp_block_until_writeable(rtsp_ctx->prvi->skt_num)<=0)
	{
		rtsp_printf("socket is unwriteable\n");
		return -2;
	}
#endif

redo:
    ret = send(rtsp_ctx->prvi->skt_num, cmd, strlen(cmd), 0);
    if ((ret < 0) && (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN))
    {
        rtsp_printf("socket send faild retry, ret=%d, errno=%d\n", ret, errno);
        goto redo;
    }
    else if ((ret <= 0) && (errno != EINTR && errno != EWOULDBLOCK && errno != EAGAIN))
    {
        ret = -1;
        rtsp_printf("socket send faild, ret=%d, errno=%d\n", ret, errno);
    }

    return ret;
}

static int rtsp_get_parameter(yc_rtsp_s *rtsp_ctx)
{
	char* cmd = NULL;
	char cmdFmt[] ="GET_PARAMETER %s%s"
        			"CSeq: %d\r\n"
        			"Session: %s\r\n"
        			"%s\r\n";
	char* readBuffer;
	char* readBuf;
	int bytesRead;
	char* firstLine;
	unsigned int responseCode;
	readBuffer = (char*)rtsp_malloc(sizeof(char)*(RTSP_READ_BUF_SIZE_MAX+1));
	if(readBuffer == NULL) return -1;
	memset(readBuffer,0,RTSP_READ_BUF_SIZE_MAX+1);

	do
	{
		if (rtsp_ctx->prvi->session_id == NULL)
		{
			rtsp_printf("No RTSP session is currently in progress\n");
			break;
		}

		cmd = (char*)rtsp_malloc(RTSP_CMD_BUF_SIZE_MAX);
		if(cmd == NULL)
		{
			rtsp_free(readBuffer);
			return -1;
		}
		memset(cmd, 0, RTSP_CMD_BUF_SIZE_MAX);
		sprintf(cmd, cmdFmt,
			rtsp_ctx->prvi->url,
			rtsp_version_str,
			++rtsp_ctx->prvi->cseq,
			rtsp_ctx->prvi->session_id,
			user_agent_str);
		rtsp_printf("\n""GET_PARAMETER command-%d:\n%s\n",rtsp_ctx->prvi->cseq,cmd);

		if (rtsp_send_quest(rtsp_ctx, cmd) <= 0)
		{
			break;
		}

		readBuf = readBuffer;
		bytesRead = rtsp_get_response(rtsp_ctx,readBuf, RTSP_READ_BUF_SIZE_MAX);

		if (bytesRead <= 0) break;

		rtsp_printf("bytesRead is %d\n",bytesRead);
		rtsp_printf("GET_PARAMETER response-%d:\n%s\n",rtsp_ctx->prvi->cseq,readBuf);

		firstLine = readBuf;
		/*char* nextLineStart =*/ rtsp_get_line(firstLine);

		if (rtsp_parse_response_code(firstLine,&responseCode)) break;

		if (responseCode != 200)
		{
			rtsp_printf("cannot handle GET_PARAMETER response\n ");
			break;
		}

        rtsp_printf("sent GET_PARAMETER Streams successful\n");

		rtsp_free(cmd);
		rtsp_free(readBuffer);
		return 0;
	} while (0);

	rtsp_free(cmd);
	rtsp_free(readBuffer);
	rtsp_printf("GET_PARAMETER Streams failed\n");
	return -1;
}

static int rtsp_request_option(yc_rtsp_s *rtsp_ctx)
{
	char* cmd = NULL;
	char* readBuffer;
	char* readBuf;
	char* firstLine;
	unsigned int responseCode;
	int bytesRead;

	readBuffer = (char *)rtsp_malloc((RTSP_READ_BUF_SIZE_MAX+1)*sizeof(char));
	if(readBuffer == NULL) return -1;

	memset(readBuffer,0,RTSP_READ_BUF_SIZE_MAX+1);
	readBuf = readBuffer;

	do
	{
		cmd = (char*)rtsp_malloc(RTSP_CMD_BUF_SIZE_MAX);
		if(cmd == NULL) return -1;
		memset(cmd, 0, RTSP_CMD_BUF_SIZE_MAX);
		sprintf(cmd, "OPTIONS %s%sCSeq: %d\r\n", rtsp_ctx->prvi->url,
                                     			 rtsp_version_str,
                                     			 ++rtsp_ctx->prvi->cseq);
        if (rtsp_ctx->prvi->session_id)
		{
		    strcat(cmd, "Session: ");
		    strcat(cmd, rtsp_ctx->prvi->session_id);
		    strcat(cmd, "\r\n");
		}
		strcat(cmd, user_agent_str);
		strcat(cmd, "\r\n");

		rtsp_printf("\n""OPTIONS command-%d:\n%s\n",rtsp_ctx->prvi->cseq,cmd);

		if (rtsp_send_quest(rtsp_ctx, cmd) <= 0)
		{
			break;
		}

		bytesRead = rtsp_get_response(rtsp_ctx,readBuf, RTSP_READ_BUF_SIZE_MAX);
		if (bytesRead <= 0) break;
		rtsp_printf("OPTIONS response-%d:\n%s\n",rtsp_ctx->prvi->cseq,readBuf);

		firstLine = readBuf;
		rtsp_get_line(firstLine);
		if (rtsp_parse_response_code(firstLine,&responseCode)) break;
		if (responseCode != 200)
		{
			rtsp_printf("cannot handle OPTIONS response\n ");
			break;
		}

		rtsp_free(readBuffer);
		rtsp_free(cmd);
		return 0;
	} while (0);

    rtsp_printf("OPTION Streams failed\n");
	rtsp_free(readBuffer);
	rtsp_free(cmd);
	return -1;
}

static int rtsp_describe_sdp(yc_rtsp_s *ctx, char** Description)
{
	int fSocketNum;
	char *readBuffer;
	char* readBuf;
    char *sdp = NULL;
	int bytesRead;
  	char* cmd = NULL;
	char* firstLine;
	unsigned int responseCode;
	unsigned short contentLength;
	unsigned int numExtraBytesNeeded;
	char* ptr;
	int bytesRead2;

	readBuffer = (char *)rtsp_malloc((RTSP_READ_BUF_SIZE_MAX+1)*sizeof(char));
	if(readBuffer == NULL)
	{
		rtsp_printf("failed to alloc the memory\n");
    		return -1;
	}
	memset(readBuffer,0,RTSP_READ_BUF_SIZE_MAX+1);
	readBuf =readBuffer;

	cmd = (char *)rtsp_malloc(RTSP_CMD_BUF_SIZE_MAX);
	if(cmd ==NULL)
	{
		rtsp_printf("failed to alloc the memory\n");
		rtsp_free(readBuffer);
		return -1;
	}

	memset(cmd, 0, RTSP_CMD_BUF_SIZE_MAX);
	fSocketNum = ctx->prvi->skt_num;
	do
	{
		sprintf(cmd, "DESCRIBE %s%sCSeq: %d\r\nAccept: application/sdp\r\n%s\r\n",
		        ctx->prvi->url, rtsp_version_str, ++ctx->prvi->cseq, user_agent_str);
		rtsp_printf("\n""DESCRIBE command-%d:\n%s\n", ctx->prvi->cseq, cmd);

  		if (rtsp_send_quest(ctx, cmd) <= 0)
		{
			rtsp_printf("DESCRIBE send() failed\n ");
			break;
		}

		bytesRead = rtsp_get_response(ctx,readBuf, RTSP_READ_BUF_SIZE_MAX);
		if (bytesRead <= 0) break;

		firstLine = readBuf;
		rtsp_get_line(firstLine);
		if (sscanf(firstLine, "%*s%u", &responseCode) != 1)
		{
			rtsp_printf("no response code in line\n");
			break;
		}
		if (responseCode != 200)
		{
			rtsp_printf("cannot handle DESCRIBE response\n");
			break;
		}

#if !RTSP_UDP
	    ctx->prvi->tmode = RTSP_TRANSPORT_MODE_TCP;
#endif

        contentLength = ctx->prvi->sdp_len;
		if(contentLength > 0)
		{
			sdp = rtsp_malloc(contentLength);
			if (NULL == sdp)
			{
				rtsp_printf("malloc sdp content failed\n");
				break;
			}
			memset(sdp, 0, contentLength);

			numExtraBytesNeeded = contentLength;// - numBodyBytes;

			bytesRead = 0;

			while (numExtraBytesNeeded > 0)
			{
				if(rtsp_block_until_readable(fSocketNum)<=0)
				{
					rtsp_printf("socket is unreadable\n");
					break;
				}

				ptr = sdp+bytesRead;

redo:
				bytesRead2 = recv(fSocketNum, (unsigned char*)ptr,numExtraBytesNeeded,0);
                if (bytesRead2 == 0)
            	{
            	    rtsp_printf("get sdp sokcet was closed.\n");
                    break;
            	}
            	else if (bytesRead2 < 0)
            	{
                    if (errno == EINTR || errno == EWOULDBLOCK || errno == EAGAIN)
                    {
                        rtsp_printf("get sdp sokcet error, continue...\n");
                        goto redo;
                    }
                    else
                    {
                        rtsp_printf("get sdp sokcet error.\n");
                        break;
                    }
            	}

				ptr[bytesRead2] = '\0';


				bytesRead += bytesRead2;
				numExtraBytesNeeded -= bytesRead2;
			}
			if (numExtraBytesNeeded > 0) break;
            *Description = sdp;
			rtsp_free(readBuffer);
			rtsp_free(cmd);
			return 0;
		}
	}while(0);

    rtsp_printf("DESCRIBE Streams failed\n");
	rtsp_free(readBuffer);
	rtsp_free(cmd);

	return -1;
}

static int rtsp_setup_session(yc_rtsp_s *rtsp_ctx)
{
	int fSocketNum = rtsp_ctx->prvi->skt_num;
	char* cmd = NULL;
	char *readBuffer;
	char* readBuf ;
	char* firstLine;
	unsigned int responseCode;
	int bytesRead;
    char temp[64];

	readBuffer = (char *)rtsp_malloc((RTSP_READ_BUF_SIZE_MAX+1)*sizeof(char));
	if(fSocketNum<0) return -1;

	if(readBuffer == NULL) return -1;
	memset(readBuffer,0,RTSP_READ_BUF_SIZE_MAX+1);
	readBuf = readBuffer;
	do
	{
		cmd = (char *)rtsp_malloc(RTSP_CMD_BUF_SIZE_MAX);
		if(cmd == NULL)
		{
			rtsp_printf("cmd failed to alloc the memory\n");
			rtsp_free(readBuffer);
			return -1;
		}

		if (rtsp_ctx->media->control_path)
		{
            if (0 == strncmp(rtsp_ctx->media->control_path, "rtsp://", 7))
            {
                strcpy(readBuf, rtsp_ctx->media->control_path);
            }
            else
            {
                strcpy(readBuf, rtsp_ctx->prvi->url);
                if ('/' != readBuf[strlen(readBuf) - 1])
                    strcat(readBuf, "/");
                strcat(readBuf, rtsp_ctx->media->control_path);
            }
		}
		else
		{
            strcpy(readBuf, rtsp_ctx->prvi->url);
		}

		memset(cmd, 0, RTSP_CMD_BUF_SIZE_MAX);
        sprintf(cmd, "SETUP %s%sCSeq: %d\r\nTransport: RTP/AVP/", readBuf,
    			rtsp_version_str, ++rtsp_ctx->prvi->cseq);

		if (RTSP_TRANSPORT_MODE_TCP == rtsp_ctx->prvi->tmode)
		{
            strcat(cmd, "TCP;unicast;");

            rtsp_ctx->prvi->rtcpNumber = rtsp_ctx->prvi->rtpNumber + 1;
            sprintf(temp, "interleaved=%u-%u", rtsp_ctx->prvi->rtpNumber,
                                                   rtsp_ctx->prvi->rtcpNumber);
            rtsp_ctx->prvi->rtpNumber += 2;
            strcat(cmd, temp);
            if (0 == strncmp(rtsp_ctx->prvi->server, "WMServer/", 9))
                strcat(cmd, ";mode=PLAY");
            strcat(cmd, "\r\n");
            if (rtsp_ctx->prvi->session_id)
            {
                strcat(cmd, "Session: ");
                strcat(cmd, rtsp_ctx->prvi->session_id);
                strcat(cmd, "\r\n");
            }
            strcat(cmd, user_agent_str);
            strcat(cmd, "\r\n");
		}
#if RTSP_UDP
		else if (RTSP_TRANSPORT_MODE_UDP == rtsp_ctx->prvi->tmode)
		{
            strcat(cmd, "UDP;unicast;");
            if (0 == rtsp_ctx->prvi->rtp_local_port)
		        rtsp_ctx->prvi->rtp_local_port = RTSP_RTP_DEFAULT_LOCAT_PORT;
            sprintf(temp, "client_port=%hu-%hu", rtsp_ctx->prvi->rtp_local_port,
    			                                 rtsp_ctx->prvi->rtp_local_port + 1);
            strcat(cmd, temp);
            if (0 == strncmp(rtsp_ctx->prvi->server, "WMServer/", 9))
                strcat(cmd, ";mode=PLAY");
            strcat(cmd, "\r\n");
            if (rtsp_ctx->prvi->session_id)
            {
                strcat(cmd, "Session: ");
                strcat(cmd, rtsp_ctx->prvi->session_id);
                strcat(cmd, "\r\n");
            }
            strcat(cmd, user_agent_str);
            strcat(cmd, "\r\n");
		}
#endif
		rtsp_printf("\n""SETUP command-%d:\n%s\n",rtsp_ctx->prvi->cseq,cmd);

		if (rtsp_send_quest(rtsp_ctx, cmd) <= 0)
		{
			rtsp_printf("SETUP send() failed\n");
			break;
		}

        memset(readBuf,0,RTSP_READ_BUF_SIZE_MAX+1);
		bytesRead = rtsp_get_response(rtsp_ctx,readBuf, RTSP_READ_BUF_SIZE_MAX);
		if (bytesRead <= 0) break;

		rtsp_printf("SETUP response-%d:\n%s\n",rtsp_ctx->prvi->cseq,readBuf);

		firstLine = readBuf;
		rtsp_get_line(firstLine);
		if (rtsp_parse_response_code(firstLine, &responseCode)) break;
		if (responseCode != 200)
		{
			rtsp_printf("cannot handle SETUP response\n");
			break;
		}

		rtsp_free(cmd);
		rtsp_free(readBuffer);

#if RTSP_UDP
    if (RTSP_TRANSPORT_MODE_UDP == rtsp_ctx->prvi->tmode)
    {
        bytesRead = rtsp_crate_rtp_connect(rtsp_ctx);
        if (bytesRead)
            goto fail;
        bytesRead = rtsp_crate_rtcp_connect(rtsp_ctx);
        if (bytesRead)
            goto fail;
    }
#endif

		return 0;
	} while (0);

    rtsp_printf("SETUP Streams failed\n");
	rtsp_free(cmd);
	rtsp_free(readBuffer);
#if RTSP_UDP
fail:
    if (-1 != rtsp_ctx->prvi->rtp_skt_num)
    {
        close(rtsp_ctx->prvi->rtp_skt_num);
        rtsp_ctx->prvi->rtp_skt_num = -1;
    }
    if (-1 != rtsp_ctx->prvi->rtcp_skt_num)
    {
        close(rtsp_ctx->prvi->rtcp_skt_num);
        rtsp_ctx->prvi->rtcp_skt_num = -1;
    }
#endif
	return -1;
}

static int rtsp_setup_streams(yc_rtsp_s *rtsp_ctx)
{
    int subsessionNum;

	subsessionNum = rtsp_ctx->prvi->session_num;
	rtsp_printf("sesssion number is %d\r\n\n", subsessionNum);
    while(subsessionNum > 0)
    {
    	if(rtsp_setup_session(rtsp_ctx))
    	{
    		rtsp_printf("Setup rtsp_media_info ID[%d] Failed\n",
    		            rtsp_ctx->prvi->session_num - subsessionNum);
    		return -1;
    	}
    	subsessionNum--;
    }
    rtsp_printf("Setup Streams successful\n");

#if RTSP_UDP
    rtsp_ctx->prvi->stat.max_seq   = 0;
    rtsp_ctx->prvi->stat.probation = 1;
#endif

    if (0 == rtsp_ctx->prvi->block_size)
        rtsp_ctx->prvi->block_size = RTSP_DEFAULT_BLOCK_SIZE;
    if (0 == rtsp_ctx->prvi->timeout)
        rtsp_ctx->prvi->timeout = RTSP_DEFAULT_ALIVE_TIME;
    if (0 == rtsp_ctx->prvi->maxps)
        rtsp_ctx->prvi->maxps = RTSP_DEFAULT_MAXPS;

    rtsp_printf("block size '%hu', timeout '%hu', maxps '%hu'\n", rtsp_ctx->prvi->block_size,
                                                            rtsp_ctx->prvi->timeout,
                                                            rtsp_ctx->prvi->maxps);

	return 0;
}

static int rtsp_play_session(yc_rtsp_s *rtsp_ctx, int start, int end)//double start, double end)
{
	char* cmd = NULL;
	char startStr[30], endStr[30];
	char *readBuffer;
	char* readBuf;
	int bytesRead;
	char* firstLine;
	unsigned int responseCode;
    char temp[128];

	readBuffer = (char*)rtsp_malloc(sizeof(char)*(RTSP_READ_BUF_SIZE_MAX+1));
	if(readBuffer == NULL) return -1;
	memset(readBuffer,0,RTSP_READ_BUF_SIZE_MAX+1);
	do
	{
		if (rtsp_ctx->prvi->session_id == NULL)
		{
			rtsp_printf("No RTSP session is currently in progress\n");
			break;
		}

        if (RTSP_STATE_PAUSE != rtsp_ctx->prvi->state)
        {
            sprintf(startStr,"%d",start);
            sprintf(endStr,"%d",end);

    		if (start==-1) startStr[0]='\0';
    		if (end == -1) endStr[0] = '\0';
    	}

		cmd = (char*)rtsp_malloc(RTSP_CMD_BUF_SIZE_MAX);
		if(cmd == NULL)
		{
			rtsp_free(readBuffer);
			return -1;
		}
		memset(cmd, 0, RTSP_CMD_BUF_SIZE_MAX);

		sprintf(cmd, "PLAY %s%sCSeq: %d\r\nSession: %s\r\n", rtsp_ctx->prvi->url,
                                                 			  rtsp_version_str,
                                                 			  ++rtsp_ctx->prvi->cseq,
                                                 			  rtsp_ctx->prvi->session_id);
        if (RTSP_STATE_PLAY == rtsp_ctx->prvi->state)
        {
            sprintf(temp, "Range: npt=%s-%s\r\n", startStr, endStr);
            strcat(cmd, temp);
        }
        strcat(cmd, user_agent_str);
        strcat(cmd, "\r\n");

		rtsp_printf("\n""PLAY command-%d:\n%s\n",rtsp_ctx->prvi->cseq,cmd);
		if (rtsp_send_quest(rtsp_ctx, cmd) <= 0)
		{
			rtsp_printf("PLAY send() failed\n ");
			break;
		}

		readBuf = readBuffer;

		bytesRead = rtsp_get_response(rtsp_ctx,readBuf, RTSP_READ_BUF_SIZE_MAX);

		if (bytesRead <= 0) break;

		rtsp_printf("PLAY response-%d:\n%s\n",rtsp_ctx->prvi->cseq,readBuf);


		firstLine = readBuf;
		rtsp_get_line(firstLine);
		if (rtsp_parse_response_code(firstLine,&responseCode)) break;
		if (responseCode != 200)
		{
			rtsp_printf("cannot handle PLAY response\n ");
			break;
		}

		rtsp_free(cmd);
	    rtsp_free(readBuffer);
	    rtsp_ctx->prvi->state = RTSP_STATE_PLAY;
		return 0;
	} while (0);

    rtsp_printf("PLAY Streams failed\n");
	rtsp_free(cmd);
    rtsp_free(readBuffer);
	return -1;
}

static int rtsp_teardown_session(yc_rtsp_s *pRtsp)
{
	char* cmd = NULL;
	char* cmdFmt = "TEARDOWN %s%s"
				"CSeq: %d\r\n"
				"Session: %s\r\n"
				"%s\r\n";
	char* readBuffer;
	char* readBuf;
	char* firstLine;
	unsigned int responseCode;
	int bytesRead;
	readBuffer = (char *)rtsp_malloc((RTSP_READ_BUF_SIZE_MAX+1)*sizeof(char));
	if(readBuffer == NULL) return -1;
	memset(readBuffer,0,RTSP_READ_BUF_SIZE_MAX+1);
	readBuf = readBuffer;
	do
	{
		if (pRtsp->prvi->session_id == NULL)
		{
			rtsp_printf("No RTSP session is currently in progress\n");
			break;
		}

		// Send the TEARDOWN command:

		// First, construct an authenticator string:

		cmd = (char*)rtsp_malloc(RTSP_CMD_BUF_SIZE_MAX);
		if(cmd == NULL) return -1;
		memset(cmd,0,RTSP_CMD_BUF_SIZE_MAX);
		sprintf(cmd, cmdFmt,
			pRtsp->prvi->url,
			rtsp_version_str,
			++pRtsp->prvi->cseq,
			pRtsp->prvi->session_id,
			user_agent_str);

		rtsp_printf("\n""TEARDOWN command-%d:\n%s\n",pRtsp->prvi->cseq,cmd);
		if (rtsp_send_quest(pRtsp, cmd) <= 0)
		{
			rtsp_printf("TEARDOWN send() failed\n ");
			break;
		}

		bytesRead = rtsp_get_response(pRtsp,readBuf, RTSP_READ_BUF_SIZE_MAX);
		if (bytesRead <= 0) break;
		rtsp_printf("TEARDOWN response-%d:\n%s\n",pRtsp->prvi->cseq,readBuf);
		// Inspect the first line to check whether it's a result code 200
		firstLine = readBuf;
		/*char* nextLineStart =*/ rtsp_get_line(firstLine);
		if (rtsp_parse_response_code(firstLine,&responseCode)) break;
		if (responseCode != 200)
		{
			rtsp_printf("cannot handle TEARDOWN response\n ");
			break;
		}

		rtsp_free(readBuffer);
		rtsp_free(cmd);
		return 0;
	} while (0);

    rtsp_printf("TEARDOWN Streams failed\n");
	rtsp_free(readBuffer);
	rtsp_free(cmd);
	return -1;
}

static void rtsp_keep_alive(void *p)
{
    int ret = -1;
    yc_rtsp_s *rtsp_ctx = (yc_rtsp_s *)p;

    if (RTSP_STATE_STOP == rtsp_ctx->prvi->state)
        return;

    if ((0 == strncmp(rtsp_ctx->prvi->server, "WMServer/", 9)) ||
        (rtsp_ctx->prvi->server_method & YC_BIT(RTSP_METHOD_GET_PARAMETER)))
    {
        if (RTSP_TRANSPORT_MODE_TCP == rtsp_ctx->prvi->tmode)
            ret = rtsp_get_parameter(rtsp_ctx);
    }
    else
    {
        ret = rtsp_request_option(rtsp_ctx);
    }

    if (ret)
        rtsp_ctx->prvi->state = RTSP_STATE_STOP;

    return;
}

#if RTSP_UDP
static int rtsp_resetup(yc_rtsp_s *rtsp_ctx)
{
    if (RTSP_TRANSPORT_MODE_UDP == rtsp_ctx->prvi->tmode)
    {
        rtsp_printf("udp timeout retry tcp\n");
        rtsp_ctx->prvi->tmode = RTSP_TRANSPORT_MODE_TCP;
        rtsp_setup_streams(rtsp_ctx);
    }

    return 0;
}
#endif

static void rtsp_dump(yc_rtsp_s *rtsp_ctx)
{
#if 1
    unsigned char i = 0;
    yc_rtsp_media_s *subsession = NULL;

    subsession = rtsp_ctx->media;
    if (NULL != subsession)
    {
        printf("\r\nStreamID: %d\r\n"
               "  Title             : %s\r\n"
               "  MediumName        : %s\r\n"
               "  CodecName         : %s\r\n"
               "  PayloadFormat     : %hhu\r\n"
               "  TimestampFrequency: %u\r\n"
               "  Duration          : %llu (ms)\r\n"
               "  Server            : %s\r\n"
               "\r\n\r\n", i,
                           rtsp_ctx->prvi->title ? rtsp_ctx->prvi->title : "",
                           subsession->media_name ? subsession->media_name : "",
                           subsession->codec_name ? subsession->codec_name : "",
                           subsession->payload_type,
                           subsession->timestamp_frequency,
                           subsession->time_len,
                           rtsp_ctx->prvi->server ? rtsp_ctx->prvi->server : "");

        subsession = subsession->next;
        i++;
    }
#endif

    return;
}

int yc_rtsp_pause(IN yc_rtsp_s *pRtsp)
{
	char* cmd = NULL;
	char cmdFmt[] =
			"PAUSE %s/ RTSP/1.0\r\n"
			"CSeq: %d\r\n"
			"Session: %s\r\n"
			"%s\r\n";
	unsigned int cmdSize;
	char* readBuffer;
	char* readBuf;
	int bytesRead;
	char* firstLine;
	unsigned int responseCode;
	readBuffer = (char*)rtsp_malloc(sizeof(char)*(RTSP_READ_BUF_SIZE_MAX+1));
	if(readBuffer == NULL) return -1;
	memset(readBuffer,0,RTSP_READ_BUF_SIZE_MAX+1);

	do
	{
		// First, make sure that we have a RTSP session in progress
		if (pRtsp->prvi->session_id == NULL)
		{
			rtsp_printf("No RTSP session is currently in progress\n");
			break;
		}

		cmdSize = strlen(cmdFmt)
			+ strlen(pRtsp->prvi->url)
			+ 20 /* max int len */
			+ strlen(pRtsp->prvi->session_id)
			+ strlen(user_agent_str);
		cmd = (char*)rtsp_malloc(sizeof(char)*cmdSize);
		if(cmd == NULL)
		{
			rtsp_free(readBuffer);
			return -1;
		}
		memset(cmd,0,cmdSize);
		sprintf(cmd, cmdFmt,
			pRtsp->prvi->url,
			++pRtsp->prvi->cseq,
			pRtsp->prvi->session_id,
			user_agent_str);
		rtsp_printf("PAUSE command-%d:\n%s\n",pRtsp->prvi->cseq,cmd);

		if (rtsp_send_quest(pRtsp, cmd) <= 0)
		{
			rtsp_printf("PAUSE send() failed!\n ");
			break;
		}

		// Get the response from the server:
		readBuf = readBuffer;
		bytesRead = rtsp_get_response(pRtsp,readBuf, RTSP_READ_BUF_SIZE_MAX);

		if (bytesRead <= 0) break;

		rtsp_printf("bytesRead is %d\n",bytesRead);
		rtsp_printf("PAUSE response-%d:\n%s\n",pRtsp->prvi->cseq,readBuf);

		firstLine = readBuf;
		/*char* nextLineStart =*/ rtsp_get_line(firstLine);

		if (rtsp_parse_response_code(firstLine,&responseCode)) break;

		if (responseCode != 200)
		{
			rtsp_printf("cannot handle PAUSE response\n ");
			break;
		}

		rtsp_free(cmd);
		rtsp_free(readBuffer);
		rtsp_printf("Pause Streams successful\n");
		pRtsp->prvi->state = RTSP_STATE_PAUSE;
		return 0;
	} while (0);

	rtsp_free(cmd);
	rtsp_free(readBuffer);
	rtsp_printf("Pause Streams failed\n");
	return -1;
}

int yc_rtsp_play(IN yc_rtsp_s *pRtsp)
{
    int ret;

    ret = rtsp_play_session(pRtsp, 0, -1);

	if(ret)
	{
		rtsp_printf("Play MediaSession failed\n");
	}
	else
	{
	    if (0 == pRtsp->prvi->scale)
	        pRtsp->prvi->scale = 1 * 1000;
#if RTSP_UDP
	    pRtsp->prvi->is_retry = 1;
#endif
	    rtsp_printf("Play MediaSession successful\n");
    }

    return ret;
}

void yc_rtsp_close(IN yc_rtsp_s *pRtsp)
{
    pRtsp->prvi->state = RTSP_STATE_STOP;
    rtsp_teardown_session(pRtsp);
    rtsp_clear_up(pRtsp);

    return;
}

yc_rtsp_s *yc_rtsp_connect(IN const char *pcUrl)
{
    int result = -1;
    char* sdpDescription = NULL;
	yc_rtsp_s *rtsp_ctx = NULL;
    yc_rtsp_priv_s *prvi = NULL;

	rtsp_ctx = rtsp_malloc(sizeof(yc_rtsp_s));
	if (NULL == rtsp_ctx)
	{
	    rtsp_printf("failed to malloc rtsp context\n");
        goto fail;
	}

	prvi = rtsp_malloc(sizeof(yc_rtsp_priv_s));
	if (NULL == prvi)
	{
	    rtsp_printf("failed to malloc rtsp priv\n");
        goto fail;
	}

	memset(rtsp_ctx, 0, sizeof(yc_rtsp_s));
	memset(prvi, 0, sizeof(yc_rtsp_priv_s));
	prvi->skt_num      = -1;
	prvi->rtp_skt_num  = -1;
	prvi->rtcp_skt_num = -1;
	INIT_LIST_HEAD(&prvi->stNode);
	result = pthread_rwlock_init(&prvi->stLock, NULL);
    if (0 != result)
    {
        rtsp_printf("%s init lock failed.", pcUrl);
    }
	rtsp_ctx->prvi = prvi;

	prvi->url = rtsp_str_dup(pcUrl);
	if (NULL  == prvi->url)
	{
	    rtsp_printf("failed to dup url\n");
        goto fail;
	}

	rtsp_printf("##start rtsp_open_connection %s\n", prvi->url);
	prvi->skt_num = rtsp_open_connection(prvi->url);
	rtsp_printf("socketNum is %d\n", prvi->skt_num);
	if(prvi->skt_num < 0)
	{
		rtsp_printf("failed to open the URL\n");
		goto fail;
	}

	result = rtsp_request_option(rtsp_ctx);
	if (result < 0)
	{
		rtsp_printf("Failed to request option\n");
		goto fail;
	}

	result = rtsp_describe_sdp(rtsp_ctx, &sdpDescription);
	if (result < 0)
	{
		rtsp_printf("Failed to get a SDP description from URL\n");
		goto fail;
	}

	rtsp_printf("SDP description:\n%s\n", sdpDescription);

	result = rtsp_sdp_init(rtsp_ctx, sdpDescription);
	rtsp_free(sdpDescription);
	if(result < 0)
	{
		rtsp_printf("Failed to initialize a SDP description\n");
		goto fail;
	}

	result = rtsp_setup_streams(rtsp_ctx);
	if(result < 0)
	{
		rtsp_printf("Failed to SETUP\n");
		goto fail;
	}

	prvi->buffer = rtsp_malloc(prvi->block_size);
	if (NULL == prvi->buffer)
    {
        rtsp_printf("Failed to buffer\n");
		goto fail;
    }

    prvi->timer = yc_timer_add(true, TIMER_UNIT_SC, prvi->timeout * 4 / 5/* 80% */,
                               rtsp_keep_alive, rtsp_ctx);
    if (YC_TIMER_ID_INVALID == prvi->timer)
    {
        rtsp_printf("Failed to timer\n");
		goto fail;
    }

	result = yc_rtsp_play(rtsp_ctx);
	if(result < 0)
	{
		rtsp_printf("Failed to PLAY\n");
		goto fail;
	}

	rtsp_dump(rtsp_ctx);

	return rtsp_ctx;

fail:
    rtsp_clear_up(rtsp_ctx);
    return NULL;
}

