#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/prctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../../include/basetype.h"
#include "../../include/list.h"
#include "../include/conf.h"
#include "../include/socket.h"
#include "../include/debug.h"
#include "../include/scm.h"
#include "../include/iwifi_config.h"

#define SCM_APP_NAME      	   		"scmd"

#define SCM_CONF_FILE_PATH     		"/etc/scmd.conf"
#define SCM_LOCK_FILE_PATH          "/var/run/scmd.pid"
#define SCM_LOCK_FILE_MODE          (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)
#define SCM_SERVICES_PATH      		"/usr/local/sbin"
#define SCM_SOCKET_FILE_PATH        "/var/tmp/scmd.socket"

#define SCM_STATE_TXT_STARTING		"starting"
#define SCM_STATE_TXT_RUNNING		"running"
#define SCM_STATE_TXT_STOPPING		"stoping"
#define SCM_STATE_TXT_STOPED		"stoped"
#define SCM_STATE_TXT_ABNORMAL		"abnormal"
#define SCM_STATE_TXT_EXITED		"exited"
#define SCM_STATE_TXT_EXITING		"exiting"

#define SCM_RETRY_CNT_MAX			10

typedef struct tagScm_Item{
    struct list_head list;
    char szServiceName[SCM_SERVICE_NAME_MAX];
    SCM_STATE_E enState;
    pid_t stPid;
	unsigned int uiRetryCnt;
}SCM_ITEM_S;

static int g_iLockFileFd = -1;
static int g_iScm_Socket = -1;
static struct list_head g_stScm_Head;

static void scm_set_state(IN SCM_ITEM_S *pstScmItem, IN SCM_STATE_E eState, IN bool_t bIsNormal)
{
    switch(eState)
    {
        case SCM_STATE_STOPED:
        {
            printf("service %s is %s%s...\r\n", pstScmItem->szServiceName,
                                                bIsNormal ? "" : SCM_STATE_TXT_ABNORMAL" ",
                                                bIsNormal ? SCM_STATE_TXT_STOPED : SCM_STATE_TXT_EXITED);
            break;
        }
        case SCM_STATE_STOPPING:
        {
            printf("service %s is %s%s...\r\n", pstScmItem->szServiceName,
                                                bIsNormal ? "" : SCM_STATE_TXT_ABNORMAL" ",
                                                bIsNormal ? SCM_STATE_TXT_STOPPING : SCM_STATE_TXT_EXITING);
            break;
        }
        case SCM_STATE_STARTING:
        {
            printf("service %s is %s...\r\n", pstScmItem->szServiceName, SCM_STATE_TXT_STARTING);
            break;
        }
        case SCM_STATE_RUNNING:
        {
            printf("service %s is %s...\r\n", pstScmItem->szServiceName, SCM_STATE_TXT_RUNNING);
            break;
        }
        default:
        {
            break;
        }
    }

    pstScmItem->enState = eState;

    return;
}

static void scm_msg_get_state_proc(IN SCM_EVENT_S *pstEvent,
  								   IN const char *pcSrcIP,
  								   IN unsigned short usSrcPort)
{
	int iRet = ERROR_FAILED;
	SCM_ITEM_S *pstScmItem = NULL;

	list_for_each_entry(pstScmItem, &g_stScm_Head, list)
	{
		if (0 == strcmp(pstEvent->szName, pstScmItem->szServiceName))
		{
			pstEvent->enState = pstScmItem->enState;
			break;
		}
	}

	iRet = udp_send(pcSrcIP, usSrcPort, pstEvent, sizeof(SCM_ITEM_S));
	if (ERROR_SUCCESS != iRet)
	{
		debug_print(DEBUG_TYPE_INFO, "failed to get %s 's state", pstEvent->szName);
	}

	return;
}

static void scm_msg_set_state_proc(IN const SCM_EVENT_S *pstEvent,
								   IN const char *pcSrcIP,
								   IN unsigned short usSrcPort)
{
	SCM_ITEM_S *pstScmItem = NULL;

	IGNOPARA(pcSrcIP);
	IGNOPARA(usSrcPort);

	list_for_each_entry(pstScmItem, &g_stScm_Head, list)
	{
		if (pstEvent->stPid == pstScmItem->stPid)
		{
			pstScmItem->enState = pstEvent->enState;
			break;
		}
	}

	return;
}

static void scm_rcv_pf(IN const char *pcSrcIP, IN unsigned short usSrcPort,
                       IN const void *pData, IN unsigned int uiDataLen)
{
	SCM_EVENT_S *pstEvent = NULL;

	DBGASSERT(NULL != pcSrcIP);
	DBGASSERT(NULL != pData);

	pstEvent = (SCM_EVENT_S *)pData;

	switch (pstEvent->enMsgType)
	{
		case SCM_MSG_GET_STATE:
		{
			scm_msg_get_state_proc(pstEvent, pcSrcIP, usSrcPort);
			break;
		}
		case SCM_MSG_SET_STATE:
		{
			scm_msg_set_state_proc(pstEvent, pcSrcIP, usSrcPort);
			break;
		}
		default:
		{
			DBGASSERT(0);
			break;
		}
	}

    return;
}

static void scm_restart_service(IN SCM_ITEM_S *pstScmItem)
{
	pid_t stPid;
	int iRet = ERROR_FAILED;
	char szAppPath[15 + SCM_SERVICE_NAME_MAX];/* 15-SCM_SERVICES_PATH lenth */

	scm_set_state(pstScmItem, SCM_STATE_STARTING, true);
	pstScmItem->uiRetryCnt++;

	stPid = fork();
	if (0 > stPid)
	{
        debug_print(DEBUG_TYPE_ERROR, "fail to create process for restart %s.", pstScmItem->szServiceName);
	}
	else if (0 == stPid)
	{
		szAppPath[0] = '\0';
		sprintf(szAppPath, "%s/%s", SCM_SERVICES_PATH, pstScmItem->szServiceName);

		iRet = prctl(PR_SET_NAME, pstScmItem->szServiceName, NULL, NULL, NULL);
		if (0 != iRet)
		{
			debug_print(DEBUG_TYPE_INFO, "fail to set service name %s.", pstScmItem->szServiceName);
		}

		iRet = execl(szAppPath, pstScmItem->szServiceName, NULL);
        if (-1 == iRet)
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to restart service %s.", pstScmItem->szServiceName);
        }
	}
	else
	{
		pstScmItem->stPid = stPid;
	}

	return;
}

static void scm_signal_pf(IN int signal)
{
    pid_t stPid;
    int iStatus;
    SCM_ITEM_S *pstScmItem = NULL;
    SCM_ITEM_S *pstScmItem_Next = NULL;

    while((stPid = waitpid(-1, &iStatus, WNOHANG)) > 0)
    {
        list_for_each_entry_safe(pstScmItem, pstScmItem_Next, &g_stScm_Head, list)
        {
            if (stPid == pstScmItem->stPid)
            {
                if (WEXITSTATUS(iStatus))
                {
                    scm_set_state(pstScmItem, SCM_STATE_STOPED, true);
					list_del(&pstScmItem->list);
                	free(pstScmItem);
                }
                else
                {
                    scm_set_state(pstScmItem, SCM_STATE_STOPED, false);
					if (pstScmItem->uiRetryCnt < SCM_RETRY_CNT_MAX)
					{
						scm_restart_service(pstScmItem);
					}
					else
					{
						debug_print(DEBUG_TYPE_INFO, "failed to restart service %s reached: %u, stoped restart.", pstScmItem->szServiceName, SCM_RETRY_CNT_MAX);
					}
                }
            }
        }
    }

    return;
}

static int scm_conf_load(void)
{
    int iTotal = 0;
    unsigned int uiCnt = 0;
    unsigned int uiSucCnt = 0;
    int iRet = ERROR_FAILED;
    SCM_ITEM_S *pstScmItem = NULL;
    char szServiceName[SCM_SERVICE_NAME_MAX];
    char szItemNmae[10];

    iRet = conf_get_key_4int(SCM_CONF_FILE_PATH, "info", "total", &iTotal);
    if (ERROR_SUCCESS != iRet)
    {
        debug_print(DEBUG_TYPE_ERROR, "fail to get total num from configure file.");
        return ERROR_FAILED;
    }

    for (uiCnt = 0; uiCnt < iTotal; uiCnt++)
    {
        szServiceName[0] = '\0';
        szItemNmae[0] = '\0';
        sprintf(szItemNmae, "prio%u", uiCnt);
        iRet = conf_get_key_4str(SCM_CONF_FILE_PATH, "item", szItemNmae, szServiceName);
        if (ERROR_SUCCESS == iRet)
        {
            pstScmItem = malloc(sizeof(SCM_ITEM_S));
            if (NULL != pstScmItem)
            {
                memset(pstScmItem, 0, sizeof(SCM_ITEM_S));
                strcpy(pstScmItem->szServiceName, szServiceName);
                pstScmItem->enState = SCM_STATE_STOPED;
                list_add_tail(&pstScmItem->list, &g_stScm_Head);
                uiSucCnt++;
            }
            else
            {
                debug_print(DEBUG_TYPE_ERROR, "fail to malloc for %s from configure file.", szItemNmae);
            }
        }
        else
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to load for %s from configure file.", szItemNmae);
        }
    }

    if (uiSucCnt != iTotal)
    {
        iRet = ERROR_FAILED;
        debug_print(DEBUG_TYPE_ERROR, "fail to load configure file.");
    }

    return iRet;
}

static void scm_conf_unload(void)
{
    SCM_ITEM_S *pstScmItem = NULL;
    SCM_ITEM_S *pstScmItem_Next = NULL;

    list_for_each_entry_safe(pstScmItem, pstScmItem_Next, &g_stScm_Head, list)
    {
        list_del(&pstScmItem->list);
        free(pstScmItem);
    }

    return;
}

static void scm_exit_pf(void)
{
    int iRet = ERROR_FAILED;

    scm_conf_unload();
    if (-1 != g_iScm_Socket)
    {
        iRet = udp_server_destroy(g_iScm_Socket);
        if (ERROR_SUCCESS != iRet)
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to destroy udp server.");
        }
        g_iScm_Socket = -1;
    }

    if (-1 != g_iLockFileFd)
    {
        close(g_iLockFileFd);
        g_iLockFileFd = -1;
    }

    printf("service %s is %s %s...\r\n", SCM_APP_NAME, SCM_STATE_TXT_ABNORMAL" ", SCM_STATE_TXT_EXITED);

    return;
}

static int scm_run_service()
{
    pid_t stPid;
    int iRet = ERROR_FAILED;
    unsigned int uiFailCnt = 0;
    SCM_ITEM_S *pstScmItem = NULL;
	char szAppPath[15 + SCM_SERVICE_NAME_MAX];/* 15-SCM_SERVICES_PATH lenth */

    list_for_each_entry(pstScmItem, &g_stScm_Head, list)
    {
        stPid = fork();
		if (0 > stPid)
		{
			uiFailCnt++;
            debug_print(DEBUG_TYPE_ERROR, "fail to create process for %s.", pstScmItem->szServiceName);
			break;
		}
        else if (0 == stPid)
        {
            scm_set_state(pstScmItem, SCM_STATE_STARTING, true);
			szAppPath[0] = '\0';
			sprintf(szAppPath, "%s/%s", SCM_SERVICES_PATH, pstScmItem->szServiceName);

			iRet = prctl(PR_SET_NAME, pstScmItem->szServiceName, NULL, NULL, NULL);
			if (0 != iRet)
			{
				debug_print(DEBUG_TYPE_INFO, "fail to set service name %s.", pstScmItem->szServiceName);
			}

            iRet = execl(szAppPath, pstScmItem->szServiceName, NULL);
            if (-1 == iRet)
            {
                uiFailCnt++;
                debug_print(DEBUG_TYPE_ERROR, "fail to run service %s.", pstScmItem->szServiceName);
				break;
            }
        }
        else
        {
            pstScmItem->stPid = stPid;
        }
    }

    if (uiFailCnt > 0)
    {
        iRet = ERROR_FAILED;
    }
    else
    {
        iRet = ERROR_SUCCESS;
    }

    return iRet;
}

static void scm_kill_service(void)
{
    int iRet = ERROR_FAILED;
    SCM_ITEM_S *pstScmItem = NULL;

    list_for_each_entry(pstScmItem, &g_stScm_Head, list)
    {
        if (0 != pstScmItem->stPid)
        {
            iRet = kill(pstScmItem->stPid, SIGKILL);
            if (0 != iRet)
            {
                debug_print(DEBUG_TYPE_ERROR, "fail to kill service %d-%s.", pstScmItem->stPid, pstScmItem->szServiceName);
            }
        }
    }

    return;
}

static int lockfile(IN int fd)
{
    struct flock fl;
    fl.l_type = F_WRLCK;
    fl.l_start = 0;
    fl.l_whence = SEEK_SET;
    fl.l_len = 0;
    return fcntl(fd, F_SETLK, &fl);
}

static bool_t scm_is_running(void)
{
    pid_t stPid;
    int iRet = ERROR_FAILED;

    g_iLockFileFd = open(SCM_LOCK_FILE_PATH, O_RDWR | O_CREAT | O_TRUNC, SCM_LOCK_FILE_MODE);
    if (0 > g_iLockFileFd)
    {
        debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: can not open lock file.");
        return true;
    }

    iRet = lockfile(g_iLockFileFd);
    if (0 != iRet)
    {
        if ((EACCES == errno) || (EAGAIN == errno))
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: do not repeat run.");
        }
        else
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: can not lock file.");
        }
        close(g_iLockFileFd);
        g_iLockFileFd = -1;
        return true;
    }

    stPid = getpid();
    write(g_iLockFileFd, &stPid, sizeof(stPid));

    return false;
}

static void scm_create_daemon(void)
{
    pid_t stPid;

    stPid=fork();
    if (stPid > 0)
    {
        exit(0);
    }
    else if (stPid < 0)
    {
        exit(1);
    }

    setsid();

    stPid=fork();
    if(stPid > 0)
    {
        exit(0);
    }
    else if(stPid < 0)
    {
        exit(1);
    }

    /* chdir("wordir"); */
    umask(0);

    return;
}

static int scm_main(IN int argc, IN char *argv[])
{
    int iRet = ERROR_FAILED;
    bool_t bIsRunning = false;

    scm_create_daemon();

    bIsRunning = scm_is_running();
    if (true == bIsRunning)
    {
        return ERROR_FAILED;
    }

    printf("\r\nservice %s is %s...\r\n", SCM_APP_NAME, SCM_STATE_TXT_STARTING);

	iRet = prctl(PR_SET_NAME, SCM_APP_NAME, NULL, NULL, NULL);
	if (0 != iRet)
	{
		debug_print(DEBUG_TYPE_INFO, "fail to set service name %s.", SCM_APP_NAME);
	}

    iRet = atexit(scm_exit_pf);
    if (0 != iRet)
    {
        debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: not register exit function.");
        return ERROR_FAILED;
    }

    /* ļǷ */
    iRet = access(SCM_CONF_FILE_PATH, F_OK);
    if (0 != iRet)
    {
        debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: configure file not exist.");
        return ERROR_FAILED;
    }

    /* ΣòǷǿưȼ */

    /* ȫscmݣ洢: ơpidstate */
    g_iScm_Socket = -1;
    INIT_LIST_HEAD(&g_stScm_Head);

    /* udp serverlib¼ */
    iRet = udp_server_create(SCM_LOCAL_PORT, scm_rcv_pf);
    if (ERROR_SUCCESS != iRet)
    {
        debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: create udp server.");
        return ERROR_FAILED;
    }

    g_iScm_Socket = iRet;

    /* עsigchldźŲ׽صûص쳣˳ӽ */
    if (SIG_ERR == signal(SIGCHLD, scm_signal_pf))
    {
        iRet = udp_server_destroy(g_iScm_Socket);
        if (ERROR_SUCCESS != iRet)
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: destroy udp server.");
        }
        g_iScm_Socket = -1;
        return ERROR_FAILED;
    }

    /* ߳: ȡļҪбforkӽ̣execбеĽ */
    iRet = scm_conf_load();
    if (ERROR_SUCCESS != iRet)
    {
        scm_conf_unload();
        iRet = udp_server_destroy(g_iScm_Socket);
        if (ERROR_SUCCESS != iRet)
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: load configure file.");
        }
        g_iScm_Socket = -1;
        return ERROR_FAILED;
    }

    iRet = scm_run_service();
    if (ERROR_SUCCESS != iRet)
    {
        scm_kill_service();
        scm_conf_unload();
        iRet = udp_server_destroy(g_iScm_Socket);
        if (ERROR_SUCCESS != iRet)
        {
            debug_print(DEBUG_TYPE_ERROR, "fail to init scmd: boot service.");
        }
        g_iScm_Socket = -1;
        return ERROR_FAILED;
    }

    printf("service %s is %s...\r\n", SCM_APP_NAME, SCM_STATE_TXT_RUNNING);
    /* ߳:  */
    for ( ; ; )
    {
        sleep(1);
    }

    return 0;
}

int main(IN int argc, IN char *argv[])
{
    int iRet = ERROR_FAILED;

    iRet = scm_main(argc, argv);

    return iRet;
}

