#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <syslog.h>

#include "yc_opt.h"
#include "yc_basetype.h"
#include "yc_debug.h"

#define DEBUG_BUFF_LEN_MAX  256

#ifdef YC_DEBUG
#define S_SYSLOG_OPTION     (LOG_CONS | LOG_PID | LOG_PERROR)
#else
#define S_SYSLOG_OPTION     (LOG_CONS | LOG_PID)
#endif

static char *debug_seclet_type(IN yc_debug_type_e enType)
{
    char *pcType = NULL;

    switch (enType)
    {
        case YC_DEBUG_TYPE_EVENT:
        {
            pcType = "Event";
            break;
        }
        case YC_DEBUG_TYPE_ERROR:
        {
            pcType = "Error";
            break;
        }
		case YC_DEBUG_TYPE_INFO:
        {
            pcType = "Info";
            break;
        }
        default:
        {
            break;
        }
    }

    return pcType;
}

#ifdef YC_DEBUG
void __yc_debug_print(IN yc_debug_type_e enType, IN const char *pcFunc, IN const char *pcFormat, ...)
{
    va_list ap;
    time_t stNowTime;
    char *pcType = NULL;
    struct tm *pstNowTime = NULL;
    char acbuf[DEBUG_BUFF_LEN_MAX];

    time(&stNowTime);
    pstNowTime = localtime(&stNowTime);

    pcType = debug_seclet_type(enType);

    acbuf[0] = '\0';

    va_start(ap, pcFormat);
    vsprintf(acbuf, pcFormat, ap);
    va_end(ap);

    printf("*%s (%02d:%02d:%02d) in %s: %s\r\n", pcType,
                                                 pstNowTime->tm_hour,
                                                 pstNowTime->tm_min,
                                                 pstNowTime->tm_sec,
                                                 pcFunc, acbuf);

    return;
}
#endif

void yc_open_log(IN const char *szIdent)
{
    openlog(szIdent, S_SYSLOG_OPTION, LOG_LOCAL2);

    return;
}

void __yc_sys_log(IN yc_debug_type_e enType, IN const char *pcFunc, IN const char *pcFormat, ...)
{
    va_list ap;
    int iPriority;
    char *pcType = NULL;
    char acbuf[DEBUG_BUFF_LEN_MAX];

    switch (enType)
    {
        case YC_DEBUG_TYPE_EVENT:
        {
            iPriority = S_SYSLOG_OPTION | LOG_NOTICE;
            break;
        }
        case YC_DEBUG_TYPE_ERROR:
        {
            iPriority = S_SYSLOG_OPTION | LOG_ERR;
            break;
        }
		case YC_DEBUG_TYPE_INFO:
        {
            iPriority = S_SYSLOG_OPTION | LOG_INFO;
            break;
        }
        default:
        {
            iPriority = S_SYSLOG_OPTION | LOG_DEBUG;
            break;
        }
    }

    pcType = debug_seclet_type(enType);

    acbuf[0] = '\0';
    va_start(ap, pcFormat);
    vsprintf(acbuf, pcFormat, ap);
    va_end(ap);

    syslog(iPriority, "*%s in %s ==> %s", pcType, pcFunc, acbuf);

    return;
}

void yc_sys_log2(IN yc_debug_type_e enType, IN const char *pcFormat, ...)
{
    va_list ap;
    int iPriority;

    switch (enType)
    {
        case YC_DEBUG_TYPE_EVENT:
        {
            iPriority = S_SYSLOG_OPTION | LOG_NOTICE;
            break;
        }
        case YC_DEBUG_TYPE_ERROR:
        {
            iPriority = S_SYSLOG_OPTION | LOG_ERR;
            break;
        }
		case YC_DEBUG_TYPE_INFO:
        {
            iPriority = S_SYSLOG_OPTION | LOG_INFO;
            break;
        }
        default:
        {
            iPriority = S_SYSLOG_OPTION | LOG_DEBUG;
            break;
        }
    }

    va_start(ap, pcFormat);
    va_end(ap);

    vsyslog(iPriority, pcFormat, ap);

    return;
}

void yc_close_log(void)
{
    closelog();

    return;
}

void yc_open_log6(IN const char *szIdent)
{
    openlog(szIdent, LOG_CONS | LOG_PID, LOG_LOCAL6);

    return;
}

void yc_sys_log6(IN const char *pcFormat, ...)
{
    va_list ap;

    va_start(ap, pcFormat);
    vsyslog(LOG_DEBUG, pcFormat, ap);
    va_end(ap);

    return;
}

void yc_sys_vlog6(IN const char *pcFormat, IN va_list vList)
{
    vsyslog(LOG_DEBUG, pcFormat, vList);

    return;
}

/* ==================== ڴ ==================== */
#if YC_OPT_MEM_DEBUG
static list_head g_stMemBlockHead        = LIST_HEAD_INIT(g_stMemBlockHead);
static pthread_rwlock_t g_stMemBlockLock = PTHREAD_RWLOCK_INITIALIZER;

typedef struct tagMem_block
{
    struct list_head stNode;
    void *ptr;
    char *func;
    int   line;
    uint  size;
}yc_mem_block_s;

void *__yc_malloc(const char *func, int line, uint size)
{
    void *ptr = NULL;
    yc_mem_block_s *mem = NULL;

    mem = malloc(sizeof(yc_mem_block_s));
    if (NULL == mem)
        return NULL;

    ptr = malloc(size);
    if (NULL != ptr)
    {
        memset(mem, 0, sizeof(yc_mem_block_s));
        INIT_LIST_HEAD(&mem->stNode);
        mem->func = func;
        mem->line = line;
        mem->size = size;
        mem->ptr  = ptr;

        pthread_rwlock_wrlock(&g_stMemBlockLock);
        list_add(&mem->stNode, &g_stMemBlockHead);
        pthread_rwlock_unlock(&g_stMemBlockLock);
    }
    else
    {
        free(mem);
    }

    return ptr;
}

void __yc_free(void *ptr)
{
    yc_mem_block_s *mem = NULL;
    yc_mem_block_s *mem_next = NULL;

    pthread_rwlock_wrlock(&g_stMemBlockLock);
    list_for_each_entry_safe(mem, mem_next, &g_stMemBlockHead, stNode)
    {
        if (ptr == mem->ptr)
        {
            list_del(mem);
            free(mem);
        }
    }
    pthread_rwlock_unlock(&g_stMemBlockLock);

    free(ptr);

    return;
}

void yc_malloc_show(void)
{
    uint uiCount = 0;
    yc_mem_block_s *mem = NULL;

    pthread_rwlock_rdlock(&g_stMemBlockLock);
    list_for_each_entry(mem, &g_stMemBlockHead, stNode)
    {
        printf("%u: [function: %s, line %d] malloc size %u.\r\n",
               ++uiCount, mem->func, mem->line, mem->size);
    }
    pthread_rwlock_unlock(&g_stMemBlockLock);

    return;
}
#endif
